/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/drapeau/PortManager/RCS/PortMgrReceiver.c,v 1.61 92/06/15 15:01:27 drapeau Exp $ */
/* $Log:	PortMgrReceiver.c,v $
 * Revision 1.61  92/06/15  15:01:27  drapeau
 * Minor change: cast two calls to "svc_getcaller()" to avoid compiler
 * warnings.  The function was not correctly defined in the RPC
 * header files (actually, it is currently defined as a macro).
 * 
 * Revision 1.6  91/09/27  17:05:43  drapeau
 * Minor change to PortMgrConnectWithPortMgr(); changed diagnostic
 * message so that it is clear exactly which application has
 * just registered.
 * 
 * Revision 1.5  91/09/18  13:11:21  drapeau
 * Added a function, PortMgrGetPortFromName(), to respond to the GetPortFromName message in the
 * MAEstro protocol.  This function is not yet fully implemented, but at this time will check for
 * matches with the "appName" field of the Port* passed in as argument.  The behavior of this new
 * function is documented in comments at the beginning of the function.
 * 
 * Also, made minor cosmetic changes to the code.
 * 
 * Revision 1.4  91/09/09  18:29:07  drapeau
 * Changed all diagnostic messages to go through the PrintDiagnostic() routine.
 * This allowed the removal of the test for the "Testing" preprocessor
 * definition.
 * 
 * Revision 1.3  91/07/31  11:06:11  drapeau
 * Modified ConnectWithPortMgr() and DisconnectFromPortMgr() as follows:
 * When an application running on the same machine as the PortManager registers
 * itself as running on machine "localhost", the PortManager's call to
 * gethostbyaddr() would return "localhost" as the official name of the caller's
 * hostname.  This is incorrect, since it causes the PortManager to register
 * the application as registering from "localhost" instead of, say, "sioux" (the
 * real host name, in other words).
 * The PortManager now asks what hostname it's running on when it first starts.
 * Whenever an application connects or disconnects with the Port Manager, the
 * Port Manager now checks to see if the calling application was on "localhost"
 * (i.e., the same host as the Port Manager).  If so, the Port Manager now
 * substitutes the real host name for the name "localhost".  Now applications
 * can register with the Port Manager on host "localhost" but the Port Manager
 * will always use the real hostname instead of the name "localhost".
 * 
 * Also, minor cosmetic code changes were made to conform to coding standards.
 * 
 * Revision 1.2  91/06/17  18:14:42  drapeau
 * Added copyright notice.
 * 
 * Revision 1.1  90/11/30  13:46:03  drapeau
 * Initial revision
 *  */

static char portMgrRecvrRcsid[] ="$Header: /Source/Media/drapeau/PortManager/RCS/PortMgrReceiver.c,v 1.61 92/06/15 15:01:27 drapeau Exp $";

#include <Receiver.h>
#include "PortList.h"
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>


extern char		portMgrHostName[256];
extern PortList*	portMgrPortList;
extern Receiver*	portMgrReceiver;
extern char		diagMessage[];
extern void		PrintDiagnostic(char*);

void PortMgrConnectWithPortMgr(Port* newApp,			    /* Warning: This function takes the implicit 2nd argument... */
			       struct svc_req* theRequest)	    /* ...passed in by the RPC layer, even though it is not... */
{								    /* ...declared in the .h file. */
  Port*			tempPtr;
  struct sockaddr_in*	callerAddr;
  struct hostent*	callingHost;
  char*			realHostName;

  callerAddr = (struct sockaddr_in*)svc_getcaller(theRequest->rq_xprt);	/* Find out where the calling app is calling from */
  callingHost = gethostbyaddr(&(callerAddr->sin_addr),		    /* Get the host name of the calling app, from its... */
			      sizeof(callerAddr->sin_addr),	    /* ...Internet address */
			      AF_INET);
  if ( strcmp(callingHost->h_name,"localhost") == 0)		    /* Was the official hostname returned as "localhost"? */
  {								    /* Yes, use the real hostname for this machine instead... */
    realHostName = (char*) portMgrHostName;			    /* ...of "localhost" */
  }
  else								    /* No, use the hostname given by the caller */
    realHostName = callingHost->h_name;
  sprintf(diagMessage,
	  "In Connect, %s is calling from %s.\n",
	  newApp->appName,
	  realHostName);
  PrintDiagnostic(diagMessage);
  tempPtr = (Port*)malloc(sizeof(Port));
  tempPtr->appName = strdup(newApp->appName);
  tempPtr->hostName = strdup(realHostName);
  tempPtr->portNumber = newApp->portNumber;
  PortListAddPortToList(&portMgrPortList, tempPtr);
  return;
}								    /* end function PortMgrConnectWithPortMgr */


void PortMgrDisconnectFromPortMgr(Port* appToRemove,		    /* Warning: This function takes the implicit 2nd argument... */
				  struct svc_req* theRequest)	    /* ...passed in by the RPC layer, even though it is not... */
{								    /* ...declared in the .h file. */
  struct sockaddr_in*	callerAddr;
  struct hostent*	callingHost;
  Port			tempPort;
  char*			realHostName;
  
  callerAddr = (struct sockaddr_in*)svc_getcaller(theRequest->rq_xprt);	/* Find out where the calling app is calling from */
  callingHost = gethostbyaddr(&(callerAddr->sin_addr),		    /* Get the host name of the calling app, from its... */
			      sizeof(callerAddr->sin_addr),	    /* ...Internet address */
			      AF_INET);
  if ( strcmp(callingHost->h_name,"localhost") == 0)		    /* Was the official hostname returned as "localhost"? */
  {								    /* Yes, use the real hostname for this machine instead... */
    realHostName = (char*) portMgrHostName;			    /* ...of "localhost" */
  }
  else								    /* No, use the hostname given by the caller */
    realHostName = callingHost->h_name;
  sprintf(diagMessage, "In Disconnect, App is calling from the computer named %s.\n",
	  realHostName);
  PrintDiagnostic(diagMessage);
  if (strcmp(appToRemove->hostName,realHostName) != 0)		    /* Did the requesting application correctly fill in its... */
    {								    /* ...hostname?  If not, do the following: */
    tempPort.appName = appToRemove->appName;			    /* Copy the requesting app's Port structure to a... */
    tempPort.portNumber = appToRemove->portNumber;		    /* ...temporary Port so as not to disturb the original */
    tempPort.hostName = realHostName;				    /* Use the calling app's real hostname, not the one given */
    PortListRemovePortFromList(&portMgrPortList,&tempPort);	    /* Remove the Port from the PortList using valid Port info */
    }
  else								    /* The requesting app correctly identified itself;... */
    PortListRemovePortFromList(&portMgrPortList,appToRemove);	    /* ...remove the Port from the PortList using the app's info */
  PortListPrint(portMgrPortList);
  return;
}								    /* end function PortMgrDisconnectFromPortMgr */


PortArray* PortMgrGetOpenApps(void* unusedArg)			    /* This function returns a list of Port's, one for each... */
{								    /* ... application currently connected to the Port Manager. */
  static PortArray	appList = {(Port*)NULL,0};
  PortList*		tempList;
  int			openAppCount;
  Port*			tempPort;
  
  openAppCount = 0;						    /* Init number of open apps to zero. */
  for (tempList = portMgrPortList; tempList;			    /* Traverse the list of open Applications kept track of... */
       tempList = tempList->next)				    /* ...by the Port Manager */
    {
      openAppCount++;						    /* One more app in the list, increment counter variable */
    }
  if (appList.portArray)
    DestroyPortArray(&appList);					    /* Free space previously allocated for a PortArray */
  appList.portArray =(Port*)malloc((openAppCount+1) *		    /* Allocate space for 1 Port per open application, plus... */
				   sizeof(Port));		    /* ...1 more for a NULL Port to end the list. */
  openAppCount = 0;						    /* Reset the counter variable for the next stmt */
  for (tempList = portMgrPortList; tempList;
       tempList = tempList->next)
    {
      tempPort = &(appList.portArray[openAppCount]);
      tempPort->appName = (char*)strdup(tempList->port->appName);
      tempPort->hostName = (char*)strdup(tempList->port->hostName);
      tempPort->portNumber = tempList->port->portNumber;
      openAppCount++;						    /* Increment the array variable counter */
    }
  bzero((char*)&appList.portArray[openAppCount],sizeof(Port));
  appList.numberOfPorts = openAppCount;
  return(&appList);						    /* Return the list of Port's */
}								    /* end function PortMgrGetOpenApps */



/******************************************************************
 *	PortMgrGetPortFromName
 *
 *	This function scans through the Port Manager's internal
 *	list of Ports (representing the currently open
 *	applications), attempting to find all matches that
 *	satisfy the criteria passed in as argument.
 *	The function takes one argument, a Port*, specifying the
 *	application to be searched for.  The Port* passed in
 *	can be filled in two different ways:
 *
 *	1) Only the "appName" field is filled in
 *	2) Both the "appName" and "hostName" field are filled in
 *	
 *	In the first case, it is assumed that the calling
 *	application is asking for any applications with the name
 *	stored in "appName"; in other words, the host on which
 *	the applications are running does not matter.  Based on
 *	this assumption, the function will do a simple string
 *	comparison to find all matching Ports, then return the
 *	resulting list of Ports (as the PortArray* return value).
 *
 *	In the second case, it is assumed that not only the name
 *	of the application is important, but also the location on
 *	the network of the desired application.
 *	This case is not currently implemented, although the
 *	function should do two string comparisons (looking at
 *	both appName and hostName fields) before returning a
 *	successful match.
 */

PortArray*	PortMgrGetPortFromName(Port* appPort)
{
  static PortArray	appList = {(Port*)NULL,0};
  PortList*		tempList;
  Port*			tempPort;
  int			matchingAppsCount;
  int			counter;
  
  if (appList.portArray)
    DestroyPortArray(&appList);					    /* Free space allocated for a PortArray from previous call */
  if (appPort == (Port*)NULL)					    /* Was the Port* passed in invalid? */
    return(&appList);						    /* Yes, return an empty PortArray* */
  matchingAppsCount = 0;					    /* Init number of matching apps to zero. */
  for (tempList = portMgrPortList; tempList;			    /* Traverse the list of open Applications kept track of... */
       tempList = tempList->next)				    /* ...by the Port Manager */
    {
      if (strcmp(tempList->port->appName, appPort->appName) == 0)   /* Was a match found on this element of the portMgrPortList? */
	matchingAppsCount++;					    /* Yes, a match was found, increment counter variable */
    }
  if (matchingAppsCount == 0)					    /* Were there any matches found? */
    return(&appList);						    /* No, return the empty PortArray and exit this function */
  appList.portArray =(Port*)malloc((matchingAppsCount+1) *	    /* Allocate space for 1 Port per open application, plus... */
				   sizeof(Port));		    /* ...1 more for a NULL Port to end the list. */
  counter = 0;							    /* Initialize counter variable for the next stmt */
  for (tempList = portMgrPortList; tempList;			    /* 2nd pass through the list: pick off matching apps... */
       tempList = tempList->next)				    /* ...and copy them to the PortArray to be returned */
  {
    if (strcmp(tempList->port->appName, appPort->appName) == 0)	    /* Was a match found on this element of the portMgrPortList? */
    {								    /* Yes, copy this Port's information into the PortArray: */
      tempPort = &(appList.portArray[counter]);			    /* Point to the appropriate element of the PortArray */
      tempPort->appName = (char*)strdup(tempList->port->appName);   /* Copy the Port information into the current element */
      tempPort->hostName = (char*)strdup(tempList->port->hostName);
      tempPort->portNumber = tempList->port->portNumber;
      counter++;						    /* Move on to next element in PortArray "appList.portArray" */
    }
  }
  bzero((char*)&appList.portArray[counter],sizeof(Port));	    /* Zero out the last element of the PortArray to be returned */
  appList.numberOfPorts = matchingAppsCount;			    /* Update number of apps that matched */
  return(&appList);
}								    /* end function PortMgrGetPortFromName */

