/******************************************************************************\
*                                                                              *
* The software for `MaMA' was developed by:    H. K. Blokhuis and              *
*                                              S. L. Voogd                     *
*                                                                              *
* as part of their graduation for:     the Hoger Informatica Onderwijs (HIO)   *
* in Enschede.                             ---------------------------------   *
* The project was carried out at:          Sun Microsystems Nederland B.V.     *
* in Amersfoort.                           -------------------------------     *
*                                                                              *
*                                                                              *
* Copyright (c) 1992 Sun Microsystems Nederland B.V.                           *
*                                                                              *
* Permission to use, copy, modify, and distribute this software and            *
* its documentation for any purpose is hereby granted without fee, provided    *
* that (i) the above copyright notices and this permission notice appear in    *
* all copies of the software and related documentation, and (ii) the name      *
* Sun may not be used in any advertising or publicity relating to              *
* the software without the specific, prior written permission of               *
* Sun.                                                                         *
*                                                                              *
* THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,           *
* EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY             *
* WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.             *
*                                                                              *
* IN NO EVENT SHALL SUN BE LIABLE FOR ANY SPECIAL, INCIDENTAL,                 *
* INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES                *
* WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT       *
* ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,        *
* ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS          *
* SOFTWARE.                                                                    *
*                                                                              *
\******************************************************************************/

/*----------------------------------------------------------------------------*\
|                                                                              |
|        File : MaMA_editlist.c                                                |
|                                                                              |
|    Contents : Functions to perform on the editlist                           |
|                                                                              |
\*----------------------------------------------------------------------------*/

#include <MaMA_include.h>
#include <MaMA_global.h>


extern ACTIONLIST *ActionList;
extern int NumberOfActions;
extern EDITLIST   *EditList;
extern EDITLIST   *CurEdit;
extern EDITLISTINFO *EditListInfo;
extern int        NumberOfEdits;
extern int        NumberOfDoc;
extern int        CurRow;
extern Xv_font    *listFont;

/******************************************************************************\
*                                                                              *
*        Name : PrintEditList                                                  *
*                                                                              *
* Description : Prints the contents of the file list                           *
*                                                                              *
\******************************************************************************/

void
PrintEditList(EDITLIST *EditListPtr)
{
    Print("--E--D--I--T--L--I--S--T--\n");
    while(EditListPtr != NIL(EDITLIST *))
    {
        sprintf(Debug_Msg, "\tlabel    : %s\n",EditListPtr->Label);
        Print(Debug_Msg);
        sprintf(Debug_Msg, "\tDur      : %d\n",EditListPtr->TotalDur);
        Print(Debug_Msg);
        sprintf(Debug_Msg, "\tselected : %d\n\n", EditListPtr->Selected);
        Print(Debug_Msg);
        EditListPtr = EditListPtr->next;
    }
    Print("--------------------------\n");
}

/******************************************************************************\
*                                                                              *
*        Name : DeleteSelectedEdit                                             *
*                                                                              *
* Description : Delete the first selected from the EditList                    *
*                                                                              *
\******************************************************************************/

EDITLIST 
*DeleteSelectedEdit(EDITLIST *EditListPtr)
{
    EDITLIST *Tmp;
    EDITLIST *ReturnPtr;

    ReturnPtr = EditListPtr;
    if (EditListPtr != NIL(EDITLIST *)) 
    { 
        if (EditListPtr->Selected)
        {
            Tmp = EditListPtr;
            ReturnPtr = EditListPtr->next; 
            free(Tmp);
        } else {
            Tmp = EditListPtr->next;
            while (Tmp != NIL(EDITLIST *))
            {    
                if (Tmp->Selected)
                {
                    EditListPtr->next = Tmp->next;
                    free(Tmp);
                    break;
                } else {
                    EditListPtr = EditListPtr->next;
                    Tmp = Tmp->next;
                } 
            }
        }
    } 
    return(ReturnPtr);
}

/******************************************************************************\
*                                                                              *
*        Name : UnselectEdit                                                   *
*                                                                              *
* Description : Deselect all edits of EditList                                 *
*                                                                              *
\******************************************************************************/

void
UnselectEdit(EDITLIST *EditListPtr)
{
    while(EditListPtr != NIL(EDITLIST *))
    {
        EditListPtr->Selected = False;
        EditListPtr = EditListPtr->next;
    }
}

/******************************************************************************\
*                                                                              *
*        Name : SelectEdit                                                     *
*                                                                              *
* Description : Set the field Selected of EditList to "Value"                  *
*               Deselect all other entries                                     *
*                                                                              *
\******************************************************************************/

void
SelectEdit(EDITLIST *EditListPtr, int row)
{
    int    entry;

    entry = 0;
    CurEdit = NIL(EDITLIST *);

    while (EditListPtr != NIL(EDITLIST *))
    {
        EditListPtr->Selected = False;

        if (entry++ == row)
        {
            EditListPtr->Selected = True;
            CurEdit = EditListPtr;
        }
        EditListPtr = EditListPtr->next;
    } 
    CurRow = row;
}

/******************************************************************************\
*                                                                              *
*        Name : InsertEdit                                                     *
*                                                                              *
* Description : Inserts a new entry in EditList                                *
*                                                                              *
\******************************************************************************/

EDITLIST 
*InsertEdit(EDITLIST *EditListPtr)
{
    EDITLIST *NewEdit;

    NewEdit = (EDITLIST *)malloc((unsigned )sizeof(EDITLIST));

    NewEdit->next = NIL(EDITLIST *);

    if (EditListPtr == NIL(EDITLIST *))
    {
        return(NewEdit);
    }

    /*
     * Place new entry at end of list
     */
    while(EditListPtr->next != NIL(EDITLIST *))
    {
        EditListPtr = EditListPtr->next;
    }
    EditListPtr->next = NewEdit;

    return(NewEdit);
}
 
/******************************************************************************\
*                                                                              *
*        Name : FillEdit                                                       *
*                                                                              *
* Description : Fill the fields of an Edit                                     *
*                                                                              *
\******************************************************************************/

void
FillEdit()
{
    int  TotalDuration; 
    int  i; 
    ACTIONLIST *ActionListPtr;
    ACTIONLIST *TmpActionListPtr;

    ActionListPtr = ActionList;

    /*
     * Calculate Total Edit Duration
     */
    TotalDuration = 0;
    for(i = 0; i < NumberOfActions ;i++)
    {
        if ((ActionListPtr->Action->ActionType == GotoNP)
          ||(ActionListPtr->Action->ActionType == GotoPP))
        {
            TotalDuration += (ActionListPtr->Duration) *
                 (ActionListPtr->Action->Param->Times); 
        } else {
            TotalDuration += (ActionListPtr->Duration);
        }
        ActionListPtr = ActionListPtr->next;
    }

    /*
     * Fill fields of Edit 
     */
    strcpy(CurEdit->Label, (char *)
        xv_get(MaMA_Edit_Panel->Edit_Label_tf, PANEL_VALUE));   
 
    CurEdit->TotalDur = TotalDuration;
    CurEdit->Selected = True;   

    TmpActionListPtr = CurEdit->ActionList;
    CurEdit->ActionList = ActionList;
    DeleteActionList(TmpActionListPtr);
    free(ActionListPtr);
    free(TmpActionListPtr);
}
 
/******************************************************************************\
*                                                                              *
*        Name : GetSelectedEdit                                                *
*                                                                              *
* Description : Return a pointer to the currently selected Edit                *
*                                                                              *
\******************************************************************************/

EDITLIST
*GetSelectedEdit(EDITLIST *EditListPtr)
{
    while (EditListPtr != NIL(EDITLIST *))
    {
        if (EditListPtr->Selected)
        {
            return(EditListPtr);
        }
        EditListPtr = EditListPtr->next;
    }
    return(NIL(EDITLIST *));
}
 
/******************************************************************************\
*                                                                              *
*        Name : SelectEditByLabel                                              *
*                                                                              *
* Description : Select the edit with the given label                           *
*                                                                              *
\******************************************************************************/

void
SelectEditByLabel(unsigned char *Label)
{
    EDITLIST *EditListPtr;
    int row;
    int i;
    int old_row;
    int LabelFound;

    row = 0;
    LabelFound = False;

    /*
     * Check only if there are Edits
     */
    if (EditList == NIL(EDITLIST*))
    {
        return;
    }

    EditListPtr = EditList;

    while (EditListPtr != NIL(EDITLIST *))
    {
        if (EditListPtr->Selected)
        {
            EditListPtr->Selected = False;
            old_row = row;
        }


        if (0 == strcmp((char *)Label, EditListPtr->Label))
        {
            LabelFound = True;
            EditListPtr->Selected = True;
            CurEdit = EditListPtr;
            /*ActionList = CurEdit->ActionList;*/

            xv_set(MaMA_Main_Window->Edit_List, PANEL_LIST_SELECT, row, 
                TRUE, NULL);
        }
        row++;
        EditListPtr = EditListPtr->next;
    }

    EditListPtr = EditList;

    if (! LabelFound)
    {
        /*
         * label name not found => restore old situation
         */
        xv_set(MaMA_Main_Window->Edit_List, PANEL_LIST_SELECT, old_row, NULL);
        EditListPtr = EditList;

        for(i = 0; i < old_row; i++)
        {
            EditListPtr = EditListPtr->next;
        }

        CurEdit = EditListPtr;
        CurEdit->Selected = True;
        /*ActionList = CurEdit->ActionList;*/
    }
}

/******************************************************************************\
*                                                                              *
*        Name : CheckLabelname                                                 *
*                                                                              *
* Description : Chop off spaces at beginning and check if Label is non-empty   *
*                                                                              *
* ReturnValue : 0 -- OK                                                        *
*               1 -- Label name is empty                                       *
*                                                                              *
\******************************************************************************/

int
CheckLabelName()
{
    EDITLIST *EditListPtr;
    char *L_Ptr;
    char Label[100];
    int i;
    int len;

    L_Ptr = (char *)xv_get(MaMA_Edit_Panel->Edit_Label_tf, PANEL_VALUE);

    /*
     * Chop of spaces at start of label
     */
    while(strncmp(L_Ptr, " ", 1) == 0)
    {
        L_Ptr++;
    }
    xv_set(MaMA_Edit_Panel->Edit_Label_tf, PANEL_VALUE, L_Ptr, NULL);

    /*
     * Check if label field is filled
     */
    if (strlen((char *)xv_get(MaMA_Edit_Panel->Edit_Label_tf,PANEL_VALUE)) == 0)
    {
        notice_prompt (MaMA_Edit_Panel->Edit_Panel, NULL,
            NOTICE_MESSAGE_STRINGS,
            dgettext("MaMA_labels", "You need to specify a label name!"),
            NULL,
            NOTICE_BUTTON, "OK", 100,
            NULL);

        return(1);
    }

    /*
     * A label name may not contain spaces -- Check it
     */
    len = strlen(L_Ptr);
    for (i = 0; i <= len; i++)
    {
        if (0 == strncmp(L_Ptr++, " ", 1))
        {
            notice_prompt (MaMA_Edit_Panel->Edit_Panel, NULL,
                NOTICE_MESSAGE_STRINGS,
                dgettext("MaMA_labels", "A label name may not contain spaces!"),
                NULL,
                NOTICE_BUTTON, "OK", 100,
                NULL);

            return(1);
        }
    }

    return(0);
}

/******************************************************************************\
*                                                                              *
*        Name : CheckIfUniqueLabel                                             *
*                                                                              *
* Description : Check if label name is unique                                  *
*                                                                              *
* ReturnValue : 0 -- OK                                                        *
*               1 -- Label name isn't unique                                   *
*                                                                              *
\******************************************************************************/
int
CheckIfUniqueLabel()
{

    EDITLIST *EditListPtr;
    char Label[100];

    /*
     * Check for unique label name
     */ 
    EditListPtr = EditList;
    strcpy(Label, (char *)xv_get(MaMA_Edit_Panel->Edit_Label_tf, PANEL_VALUE));

    while ((EditListPtr != NIL(EDITLIST *)) 
           && !(ChangeEditMode && (CurEdit == EditListPtr)))
    {   
        if (strcmp(EditListPtr->Label,Label) == 0)
        {
            notice_prompt (MaMA_Edit_Panel->Edit_Panel, NULL,
               NOTICE_MESSAGE_STRINGS,
               dgettext("MaMA_labels", "Edit labels must be unique!"),
               NULL,
               NOTICE_BUTTON, "OK", 100,
               NULL);
            return(1);
        }
        EditListPtr = EditListPtr->next;
    }
    
    return(0);
}

/******************************************************************************\
*                                                                              *
*        Name : BuilEditString                                                 *
*                                                                              *
* Description : Build the String for the GUI                                   *
*                                                                              *
* ReturnValue : The EditList String                                            *
*                                                                              *
\******************************************************************************/

char
*BuildEditString()
{
    int  sec;
    int  min;

    char sec_string[3]; 
    char min_string[3]; 
    char num_string[4]; 
    char T_String[6]; 
    char ListString[51];

    /*
     * Build the List String
     */
    strncpy(ListString, CurEdit->Label, 26);
    strncat(ListString, "                                           ", 
        (31 - strlen(ListString)));

    itoa(NumberOfActions, num_string);
    strcat(ListString, num_string);
    strncat(ListString, "          ", 7-strlen(num_string));

    min = (CurEdit->TotalDur / 60); 
    sec = (CurEdit->TotalDur % 60);
    itoa(min, min_string);
    itoa(sec, sec_string);
 
    if (min < 10)
    {
        strcpy(T_String, "0");
        strcat(T_String, min_string);
    } else {
        strcpy(T_String, min_string);
    }
    strcat(T_String,":");
    if (sec < 10)
    {
        strcat(T_String, "0");
    }
    strcat(T_String,sec_string);
    strcat(ListString, T_String);
    return(ListString);
} 

/******************************************************************************\
*                                                                              *
*        Name : FreeEditList                                                   *
*                                                                              *
* Description : Frees the allocated space for the EditList                     *
*                                                                              *
\******************************************************************************/

void
FreeEditList(EDITLIST *EditListPtr)
{
    EDITLIST *Tmp;

    if (EditListPtr == NIL(EDITLIST *))
    {
        return;
    }

    while (EditListPtr != NIL(EDITLIST *))
    {
        Tmp = EditListPtr;
        EditListPtr = EditListPtr->next;
        FreeEdit(Tmp);
    }
}

/******************************************************************************\
*                                                                              *
*        Name : FreeEdit                                                       *
*                                                                              *
* Description : Frees the allocated space for the Edit                         *
*                                                                              *
\******************************************************************************/

void
FreeEdit(EDITLIST *Edit)
{
    DeleteActionList(Edit->ActionList);
    free(Edit);
}

/******************************************************************************\
*                                                                              *
*        Name : LoadEditList                                                   *
*                                                                              *
* Description : Reads a file, containing an editlist, into the GUI             *
*                                                                              *
\******************************************************************************/

void
LoadEditList(char *FullFileName)
{
    EDITLIST *EditListPtr;
    EDITLISTINFO *EditListInfo_Old;
    ACTIONLIST *ActionListPtr;

    EditListInfo_Old = (EDITLISTINFO *)malloc((unsigned)sizeof(EDITLISTINFO));
    EditListInfo_Old->EditList = EditListInfo->EditList;
    strcpy(EditListInfo_Old->FullFileName, EditListInfo->FullFileName);

    /*
     * Read in file
     */
    if ((EditListInfo = ReadEditListFromFile(FullFileName))
        == NIL(EDITLISTINFO *))
    { 
        EditListInfo = (EDITLISTINFO *)malloc((unsigned)(sizeof(EDITLISTINFO)));
        EditListInfo->EditList = EditListInfo_Old->EditList;
        strcpy(EditListInfo->FullFileName, EditListInfo_Old->FullFileName);
        return;
    }
      
    FreeEditList(EditListInfo_Old->EditList);
    free(EditListInfo_Old);

    /*
     * Fill in filename
     */
    strcpy(EditListInfo->FullFileName, FullFileName);

    /*
     * Clear EditList
     */
    xv_set(MaMA_Main_Window->Edit_List, PANEL_LIST_DELETE_ROWS, 0,
        NumberOfEdits, NULL);
    
    EditList = EditListInfo->EditList;
 
    EditListPtr = EditListInfo->EditList;
 
    NumberOfEdits = 0;

    while(EditListPtr != NIL(EDITLIST *))
    {
        NumberOfActions = 0;
        ActionListPtr = EditListPtr->ActionList;
        while (ActionListPtr != NIL(ACTIONLIST *))
        {
            NumberOfActions++;
            ActionListPtr = ActionListPtr->next;
        }
       
    CurEdit = EditListPtr;

        xv_set(MaMA_Main_Window->Edit_List,
            PANEL_LIST_INSERT,NumberOfEdits,
            PANEL_LIST_STRING,NumberOfEdits,
            BuildEditString(),
            PANEL_LIST_FONT, NumberOfEdits, listFont,NULL);
 
        NumberOfEdits++;
        EditListPtr = EditListPtr->next;
    }
    SetHeader(FullFileName);

    if (NumberOfEdits > 0)
    {
        xv_set(MaMA_Main_Window->edit_drop_target, PANEL_DROP_FULL,
              TRUE, NULL);
    }

    
    /*
     * Mark First entry as selected
     */
    (void )SelectEdit(EditList, 0);

    if((int)xv_get(MaMA_Edit_Panel->Edit_Panel, XV_SHOW))
    {
          FillPopup();
    }
} 

/******************************************************************************\
*                                                                              *
*        Name : SwitchEdit                                                     *
*                                                                              *
* Description : Change the order of the Edits in the list by swapping          *
*               positions of EditPtr and EditPtr->next                         *
*                                                                              *
\******************************************************************************/

EDITLIST
* SwitchEdit(EDITLIST *EditPtr)
{
    EDITLIST *Next;

    Next = EditPtr->next;
    EditPtr->next = Next->next;
    Next->next = EditPtr;

    return(Next);
}

/******************************************************************************\
*                                                                              *
*        Name : MoveEditUp                                                     *
*                                                                              *
* Description : Move Edit indicated by EditPtr one place up in the list        *
*                                                                              *
\******************************************************************************/

EDITLIST
*MoveEditUp(EDITLIST *EditPtr)
{
    if ( (EditPtr == NIL(EDITLIST *))
         ||
         (EditPtr->next == NIL(EDITLIST *)))
    {
        return(EditPtr);
    }

    if ((EditPtr->next)->Selected)
    {
        return(SwitchEdit(EditPtr));
    }

    EditPtr->next = MoveEditUp(EditPtr->next);

    return(EditPtr);
}


/******************************************************************************\
*                                                                              *
*        Name : MoveEditDown                                                   *
*                                                                              *
* Description : Move Edit indicated by EditPtr one place down in the list      *
*                                                                              *
\******************************************************************************/

EDITLIST
*MoveEditDown(EDITLIST *EditPtr)
{
    if ( (EditPtr == NIL(EDITLIST *))
         ||
         (EditPtr->next == NIL(EDITLIST *)))
    {
        return(EditPtr);
    }

    if (EditPtr->Selected)
    {
        return(SwitchEdit(EditPtr));
    }

    EditPtr->next = MoveEditDown(EditPtr->next);

    return(EditPtr);
}

