/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/DTR/RCS/display.c,v 1.0 92/01/06 17:43:03 drapeau Exp Locker: derek $ */
/* $Log:	display.c,v $
 * Revision 1.0  92/01/06  17:43:03  drapeau
 * Made a number of cosmetic changes to make code easier to read and
 * to conform to programming specifications.
 * 
 * Revision 0.25  91/09/24  16:49:10  derek
 * Zoom level's max value is changed from 50 to 4.
 * 
 * Revision 0.24  91/09/18  22:47:16  derek
 * The following things are done:
 * 1.	The Makefile is changed corresponding to the changes in collab/.
 * 2.	Copyright messages included.
 * 3.	Some minor bugs fixed.
 * 
 * Revision 0.23  91/08/21  17:46:58  derek
 * The big canvas "mis-placed" bug is fixed.
 * 
 * Revision 0.22  91/08/21  11:34:09  derek
 * The following changes are made:
 * 1.	Now the duration and size of the recorded sound will be displayed
 * 	during recording.
 * 2.	I have changed GetSelection() corresponding to the request of Tek joo
 * 3.	Info Panel is added to the application.
 * 4.	Fixed SizeToFitHandler() so that when no file or buffer is currently
 * 	loaded, it would not do anything (except giving a warning
 * 	notice_prompt).
 * 5.	Inplemented the `Close' Menu option in the document menu.
 * 6.	Fixed the bug in which after ClearAll and I press PreviewEdit,
 * 	the edit wont be played.
 * 7.	I have made the changes corresponding to the change in OpenPanel's
 * 	name.  (from OpenPanel to Browse).
 * 8.	Incorporated BrowseCheck to check command line arg.
 * 9.	Changed most EditingStatusMessages to NoticePrompts.
 * 10.	SoundFileSaveAsPopUp and EditListSaveAsPopUp are removed 
 * 	from the application.
 * 
 * Revision 0.21  91/08/16  18:08:38  derek
 * 
 * The following things are done:
 * 1.	I have fixed an openpanel bug in my code in which I 
 * 	made the app return the wrong values to OpenHandler and
 * 	the SaveHandler.
 * 2.	The flashing color of the play button has been changed from
 * 	Red to Green.
 * 3.	Fixed a quantization bug: Buffer.play.end, when converted
 * 	from endingTimeInSec, should not exceed Buffer_hdr_data_size - 1.
 * 
 * Revision 0.20  91/08/14  18:43:46  derek
 * I fixed a UpdateSpaceOnTmp() bug.
 * 
 * Revision 0.19  91/08/14  18:14:49  derek
 * The following things are done:
 * 1.	I changed the name of the "save as" and "save" menu options
 * 	to be more specific.  ie. to "save sound file" and "save edit list".
 * 2.	I removed the preview panel popup and replaced it with a preview
 * 	edit button.
 * 3.	I changed the message displayed by UpdateSpaceOnTmp() to consist
 * 	only of the time remaining and the amount of space left.  There is
 * 	no more percentage indicating the capacity of /usr/tmp.
 * 
 * Revision 0.18  91/08/13  20:38:21  derek
 * The buttons (play, pause, record) will now flash after they are pressed.
 * This only applies to times when audio files (not edit-lists) are 
 * played.
 * 
 * Revision 0.17  91/08/08  11:02:02  derek
 * This is a cleaner version.  I have removed lots of printf/fprintf 
 * statements from it, and have also cleaned up the code using xsaber.
 * This version should run substantially faster.
 * 
 * Revision 0.16  91/08/07  16:23:59  derek
 * The Edit list part of DTR is done.  OpenPanel is also incorporated.
 * 
 * Revision 0.15  91/07/31  14:50:54  derek
 * I have changed the user interface so that text fields like size and
 * duration will be displayed in normal (not bold) format.   I also fixed
 * a network bug.  In addition, I have added a routine to check how
 * much disk space is left in /usr/tmp.
 * 
 * Revision 0.14  91/07/29  15:10:10  derek
 * The playing w/o stopping error is fixed.
 * 
 * Revision 0.13  91/06/26  15:55:00  derek
 * I have reformatted the code to conform coding specs.
 * 
 * Revision 0.12  91/06/25  11:38:04  derek
 * I have added the new protocol items.
 * 
 * Revision 0.11  91/06/20  19:55:17  derek
 * The network part should be working.  Also fixed numerous minor parts
 * involving the canvas and the display.
 *  */
static char rcsid[] = "$Header: /Source/Media/collab/DTR/RCS/display.c,v 1.0 92/01/06 17:43:03 drapeau Exp Locker: derek $";

#include "dtr.h"
#include "dtr_ui.h"

extern dtr_mainWindow_objects*		dtr_mainWindow;
extern dtr_editListPanelPopUp_objects*	dtr_editListPanelPopUp;

void
  InitDisplay()
{
  xv_set(dtr_mainWindow->playGain, PANEL_VALUE,			    /* Set the Play and Record Slider volume, and also the... */
	 Audio_state.play.gain, NULL);				    /* ...editListVolume Slider. */
  xv_set(dtr_editListPanelPopUp->editListVolumeSlider,
	 PANEL_VALUE, Audio_state.play.gain, NULL);
  xv_set(dtr_mainWindow->recordGain, PANEL_VALUE, 
	 Audio_state.record.gain, NULL);
  BlackColorIndex = gcm_color_index("black");
  RedColorIndex   = gcm_color_index("red");
  GreenColorIndex = gcm_color_index("green");
  UpdateSpaceOnTmp();
}

/*
 *  Show the duration of the whole sound buffer.
 */
void
  DisplaySoundDuration()
{
  double   timeInSec;
  int      minute;
  double   second;
  int      size;
  
  size = (int) Buffer.hdr.data_size;
  if (size < 0)  
  {
    fprintf(stderr, "Error:  Negative selection size.\n");
    exit(0);
  }
  timeInSec = Buffer.hdr.data_size / (double) DEVICE_SAMPLE_RATE;
  minute = irint(floor(timeInSec / 60.0));
  second = timeInSec - (60.0 * minute);
  sprintf(msg, " %d min %.2lf sec.", minute, second);
  Alert(dtr_mainWindow->durationMessage, msg);
}

/*
 *  Show the duration of the selected (highlighted) area.
 */
void
  DisplaySoundSelectionDuration()
{
  double   timeInSec;
  int      minute;
  double   second;
  int      size;
  
  size = (int) Buffer.play.end - (int) Buffer.play.start;
  if (size < 0)  
  {
    fprintf(stderr, "Error:  Negative selection size.\n");
    exit(0);
  }
  timeInSec = size / (double) DEVICE_SAMPLE_RATE;
  minute = irint(floor(timeInSec / 60.0));
  second = timeInSec - (60.0 * minute);
  sprintf(msg, "   %d min  %.2lf sec.", minute, second);
  Alert(dtr_mainWindow->selectionDurationMessage, msg);
}


/*
 *  Show the size of the whole buffer.
 */
void
  DisplaySoundSize()
{
  int    size;
  
  size = (int) Buffer.hdr.data_size;
  if (size < 0)  
  {
    fprintf(stderr, "Error:  Negative selection size");
    exit(0);
  }
  if (size > 100000) 
  {
    sprintf(msg, " %d bytes  (%.1f MB)", size, size/1000000.0);
  }
  else 
  {
    sprintf(msg, "  %d bytes  (%.1f KB)", size, size/1000.0);
  }
  Alert(dtr_mainWindow->sizeMessage, msg);
}


/*
 *  Show the size of the selected (highlighted) area.
 */
void
  DisplaySoundSelectionSize()
{
  int    size;
  
  size = (int) Buffer.play.end - (int) Buffer.play.start + 1;
  if (size < 0)  
  {
    fprintf(stderr, "Error:  Negative selection size");
    exit(0);
  }
  if (size > 100000) 
  {
    sprintf(msg, "%.1f MB", size/1000000.0);
  }
  else 
  {
    sprintf(msg, "%.1f KB", size/1000.0);
  }
  Alert(dtr_mainWindow->selectionSizeMessage, msg);
}


/*
 * Update file parameters on screen (due to the loading of a new file).
 */
void
  UpdateMessageDisplay()
{
  DisplaySoundSize();
  DisplaySoundSelectionSize();
  DisplaySoundDuration();
  DisplaySoundSelectionDuration();
}


/*
 *  Update both the zoom level scrollbar and the zoom level text field.
 */
void
  UpdateZoomLevelDisplay()
{
  xv_set(dtr_mainWindow->zoomSlider, PANEL_VALUE, 
	 irint(Zoom * 100.0), NULL);
  sprintf(msg, "%.2lf", Zoom);
  xv_set(dtr_mainWindow->zoomLevelTextField, 
	 PANEL_VALUE, msg, NULL);
}


void
  UpdateSpaceOnTmp()
{
  struct statfs *statfsbuf;
  int           totalsecs;
  int           min;
  double        sec;
  int           avail;
  int           availPercent;
  
  statfsbuf = (struct statfs *) malloc(sizeof(struct statfs));
  
  if (statfs("/usr/tmp", statfsbuf) == -1)
  {
    fprintf(stderr, "Can't do statfs on /usr/tmp.\n");
    exit(0);
  }

  avail = statfsbuf->f_bsize * statfsbuf->f_bavail  / 1000;	    
  totalsecs = irint(avail * 1000.0 / ((double) DEVICE_SAMPLE_RATE));
  min = irint(floor(totalsecs / 60.0));
  sec = totalsecs - (60.0 * min);
  
  if (avail / 1000.0 > 1.0)
  {
    sprintf(msg, "  %d min %.1lf sec.  (%.3lf MB)", min, sec, avail/1000.0);
  }
  else
  {
    sprintf(msg, "  %d min %.1lf sec.  (%d KB)", min, sec, avail);
  }
  
  xv_set(dtr_mainWindow->spaceLeftOnTmpMessage, PANEL_LABEL_STRING, msg, NULL);
}

  
void
  UpdateButtonGlow()
{
  static short count = 0;
  
  if (count == 0)
  {
    count = 1;
    if (ActiveFlag & PLAY)
    {
      xv_set(dtr_mainWindow->playButton, PANEL_ITEM_COLOR, GreenColorIndex, NULL);
    }
    else if (ActiveFlag & RECORD)
    {
      xv_set(dtr_mainWindow->recordButton, PANEL_ITEM_COLOR, RedColorIndex, NULL);
    }

  }
  else
  {
    count = 0;
    if (ActiveFlag & PLAY)
    {
      xv_set(dtr_mainWindow->playButton, PANEL_ITEM_COLOR, BlackColorIndex, NULL);
    }
    else if (ActiveFlag & RECORD)
    {
      xv_set(dtr_mainWindow->recordButton, PANEL_ITEM_COLOR, BlackColorIndex, NULL);
    }

  }
}    


void
  CancelButtonGlow()
{
  xv_set(dtr_mainWindow->playButton, PANEL_ITEM_COLOR, BlackColorIndex, NULL);
  xv_set(dtr_mainWindow->recordButton, PANEL_ITEM_COLOR, BlackColorIndex, NULL);
  xv_set(dtr_mainWindow->pauseButton, PANEL_ITEM_COLOR, BlackColorIndex, NULL);
}  


void
  UpdateRecordingMessageDisplay()
{
  UpdateSpaceOnTmp();
  DisplaySoundSize();
  DisplaySoundDuration();
}
