/*******************************************************************************
+
+  LEDA  3.0
+
+
+  avl_tree.h
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/


#ifndef LEDA_AVL_TREE_H
#define LEDA_AVL_TREE_H

//------------------------------------------------------------------------------
//
//
// Roman Schleich             (1991)
//
// Implementation follows
// Niklaus Wirth:  "Algorithmen und Datenstrukturen ",section IV.4.4.6
//
//
//------------------------------------------------------------------------------

#include <LEDA/basic.h>

//------------------------------------------------------------------------------
// some declarations and type definitions
//------------------------------------------------------------------------------

class avl_tree;
class avl_tree_node;

typedef avl_tree_node* avl_tree_node_ptr;
typedef avl_tree_node* avl_tree_item;

typedef struct liste* z_liste;

struct liste {
avl_tree_node* inhalt;
z_liste next;

LEDA_MEMORY(liste)

};

//----------------------------------------------------------------------------
//class avl_tree_node
//----------------------------------------------------------------------------

class avl_tree_node
{
  GenPtr i;                       // information
  GenPtr k;                        // searchkey

  int bal;                    //bal gibt Hoehenunterschied des linken und rechten Teilbaumes an
                              // bal = Hoehe(right)-Hoehe(left) und liegt immer zwischen -1..1 (AVL-Kr
  avl_tree_node* left;          // leftson
  avl_tree_node* right;         // rightson

  friend class avl_tree;

  public:

  GenPtr info()    { return i;}
  GenPtr key()   { return k;}
  int balance() { return bal;}

  avl_tree_node (GenPtr K=0 , GenPtr I=0 , int b=0 , 
                 avl_tree_node* leftson=0 , avl_tree_node *rightson=0)
  {
  i=I;
  k=K;
  bal=b;
  left=leftson;
  right=rightson;
  }

  LEDA_MEMORY(avl_tree_node)
};

//------------------------------------------------------------------------------
//class avl_tree
//------------------------------------------------------------------------------

class avl_tree
{
  static z_liste listpointer;
  static z_liste head;

  avl_tree_node* root;
  int count;


  avl_tree_node* copy_tree(avl_tree_node_ptr);
  avl_tree_node* del_element(avl_tree_node_ptr& ,int&);
  void del_search(GenPtr ,avl_tree_node_ptr& ,int& );
  void balance_left(avl_tree_node_ptr& ,int& );
  void balance_right(avl_tree_node_ptr& ,int& );

  avl_tree_node* ins_element(GenPtr , GenPtr , avl_tree_node_ptr& , int& );

  avl_tree_node* search(GenPtr) const;

  void fillup_liste(avl_tree_node*)const;
  void del_tree();


  virtual void clear_key(GenPtr& x) const { Clear(x); }
  virtual void clear_inf(GenPtr& x) const { Clear(x); }

  virtual void copy_key(GenPtr& x) const { Copy(x); }
  virtual void copy_inf(GenPtr& x) const { Copy(x); }

  virtual int cmp(GenPtr x,GenPtr y) const { return compare(x,y); }

  virtual int int_type() const { return 0; }

  protected:

  avl_tree_node* item(GenPtr p) const { return (avl_tree_node*)p; }

  public:

  avl_tree_node* first_item() const;
  avl_tree_node* next_item(avl_tree_node*) const;

  avl_tree_node* min() const;
  avl_tree_node* find_min() const{ return min(); }
  avl_tree_node* max() const;

  void del_min() { if (root) del(min()->k); }
  void del_item(avl_tree_node* p) { if (p) del(p->k); }
  void del_max() { if (root) del(max()->k); }

  void decrease_key(avl_tree_node* p,GenPtr k)  
  { GenPtr in=p->i;
    copy_inf(in);
    del(p->k);
    insert(k,in);
    clear_inf(in);
   }

  void del(GenPtr);
  void change_inf(avl_tree_node* , GenPtr);

  avl_tree_node* insert(GenPtr,GenPtr);

  avl_tree_node* lookup(GenPtr k) const { return search(k); }

  bool member(GenPtr k) const { return (search(k)) ? true : false; }

  GenPtr& access(GenPtr k) { return lookup(k)->i; }


  GenPtr    key(avl_tree_node* p)  const { return p->k; }
  GenPtr&   info(avl_tree_node* p) const { return p->i; }
  GenPtr    inf(avl_tree_node* p)  const { return p->i; }

  int balance(avl_tree_node* p) const { return p->bal; }
  int size()  const   { return count ; }
  int empty() const   { return root ? 0 : 1 ; }

  void clear() { del_tree(); }


  avl_tree()
  { listpointer = 0;
    head =0;
    root=0;
    count=0;
  }

   avl_tree(const avl_tree&);
   avl_tree& operator=(const avl_tree&);

   virtual ~avl_tree()  { clear(); }

};

#endif
