/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


/**********************************************************************
*
*  File: graphgen.c
*
*  Purpose: Generates triangles with normals and colors for   
*           feeding to device-specific graphing routines.
*
*/

#include "include.h"

/*******************************************************************************
*
*  Function: graphgen()
*
*  purpose:  Generates data for each triangle and calls the display
*            function graph_facet().
*
*
*  Return value: number of triangles plotted
*/

int graphgen()
{
  int  graphcount = 0;  /* number of facets done */
  int b;
  REAL *c;
  FILE *mapfd = NULL;

  (*graph_start)();  /* device-specific initialization */

  breakflag = 0;
  iterate_flag = 2;

  if ( colorflag )
    {
       if ( strlen(cmapname) == 0 )
         { prompt("Enter name of colormap file: ",cmapname);
         }
       mapfd = path_open(cmapname);
       if ( mapfd )
	{
          colormap = (maprow *)temp_calloc(4*web.bodycount,sizeof(REAL));
          for ( b = 0 ; b < web.bodycount ; b++ )
           { c = colormap[b];
             if ( fscanf(mapfd,"%lf %lf %lf %lf",c,c+1,c+2,c+3) != 4 ) break;
           }
          if ( b < web.bodycount )
           { sprintf(errmsg,
            "Colormap file has only %d entries for %d bodies.\n",
             b,web.bodycount);
	     error(errmsg,WARNING);
             for ( ; b < web.bodycount ; b++ )
              { c = colormap[b];
                c[0] = c[1] = c[2] = c[3] = 0.5;
              } 
           }
       }
   }
  /* call appropriate facet generator */
  if ( web.torus_body_flag )
    { if ( web.dimension == STRING ) torus_cells();
      else torus_bodies();
    }
  else 
    { if ( web.dimension == STRING ) plain_edges();
      else if ( web.dimension > SOAPFILM ) hi_dim_graph();
      else 
      { plain_facets();  bare_edges();
	if ( triple_edgeshow_flag ) triple_edges();
       }
    }


  /* bounding box (torus only) */
  if ( box_flag && web.torus_flag )
    { int i,j,k,m;
      REAL p[MAXCOORD];
      struct graphdata gdata[2];
      gdata[0].color = BLUE;
      for (  p[0] = 0.0 ; p[0] < 1.1 ; p[0] += 1.0 )
        for (  p[1] = 0.0 ; p[1] < 1.1 ; p[1] += 1.0 )
	 { for ( m = 0 ; m < web.sdim ; m++ )
	    { gdata[0].x[m] = 0.0;
  	      for ( k = 0 ; k < 2 ; k++ )
	        gdata[0].x[m] += p[k]*web.torus_period[k][m];
	    }
	   for ( j = 0 ; j <2 ; j++ )
             { if ( p[j] < 0.9 )
	         { p[j] += 1.0;
	           for ( m = 0 ; m < web.sdim ; m++ )
	              { gdata[1].x[m] = 0.0;
  	                for ( k = 0 ; k < 2 ; k++ )
	                  gdata[1].x[m] += p[k]*web.torus_period[k][m];
                      }
                   (*graph_edge)(gdata);
	           p[j] -= 1.0;
                 }
             }
         }
    }

  (*graph_end)();  /* device-specific termination */

  if ( colorflag ) temp_free((char *)colormap);

  return  graphcount;
}

/*****************************************************************
*
*  function: plain_facets()
*
*  purpose:  plots all facets one by one.
*
*/

void plain_facets()
{
  int i,j;
  facetedge_id fe;
  facet_id f_id;
  body_id b0_id,b1_id;
  REAL *verts[MAXCOORD+1]; /* for adjusted triangle vertices */
  struct graphdata gdata[MAXCOORD+1];

  for ( i = 0 ; i <= web.dimension ; i++ ) 
     verts[i] = gdata[i].x;
    
  FOR_ALL_FACETS(f_id)
    { 
      int nbrs;  /* number of neighboring bodies */

      if ( breakflag ) break;

      if ( (get_fattr(f_id) & (BOUNDARY|CONSTRAINT)) && !bdry_showflag )
         continue;
      if ( get_fattr(f_id) & NODISPLAY )
         continue;
      if ( get_facet_color(f_id) == CLEAR ) continue;
      gdata[0].color = get_facet_color(f_id);

      if ( no_wall_flag )
        { /* skip facets with all three vertices on walls */
          facetedge_id fe = get_facet_fe(f_id);

          if ( get_vattr(get_fe_headv(fe)) & (HIT_WALL|CONSTRAINT) )
            if ( get_vattr(get_fe_tailv(fe)) & (HIT_WALL|CONSTRAINT) ) 
              { fe = get_next_edge(fe);
                if ( get_vattr(get_fe_headv(fe)) & (HIT_WALL|CONSTRAINT) )
                  continue;
              }
        }     
      q_id = f_id;
      if ( show_expr.root && !eval(&show_expr,NULL) ) continue;

      if ( !valid_id(get_facet_body(f_id)) ) 
        f_id = facet_inverse(f_id);

      nbrs =  (valid_id(get_facet_body(f_id)) ? 1 : 0) 
                  + (valid_id(get_facet_body(facet_inverse(f_id))) ? 1 : 0);
      if ( (nbrs >= 2) && !innerflag ) continue;
      if ( (nbrs < 2) && !outerflag ) continue;
      if ( colorflag ) /* get vertex color */
        { 
          b0_id = get_facet_body(f_id);
          b1_id = get_facet_body(facet_inverse(f_id));
          for ( i = 0 ; i < FACET_VERTS ; i++ )   /* vertex loop */
            if ( !valid_id(b0_id) && !valid_id(b1_id) ) 
              gdata[i].color = 0;
            else if ( valid_id(b0_id) && !valid_id(b1_id) ) 
              gdata[i].color = ordinal(b0_id);
            else 
              gdata[i].color = ordinal(b1_id);
        }
          
      /* get vertices */
      get_facet_verts(f_id,verts,NULL);
      for ( i = 0 ; i <= web.dimension ; i++ ) 
        verts[i][HOMDIM-1] = 1.0;  /* homogeneous coord */

      /* do iner clipping, if called for */
      if ( inner_clip_flag )
        { 
          for ( i = 0 ; i < FACET_VERTS ; i++ )
            { double dist = 0.0;
              REAL *x = get_coord(web.zoom_v);

              for ( j = 0 ; j < web.sdim ; j++ )
                dist += (x[j]-verts[i][j])*(x[j]-verts[i][j]);

              if ( sqrt(dist) > inner_clip_rad ) break; /* it's a keeper */
            }
          if ( i == FACET_VERTS ) continue; /* entirely inside */
        }

      if ( !web.simplex_flag )
      {
       fe = get_facet_fe(f_id);
       for ( i = 0 ; i < FACET_VERTS ; i++ )
        { 
          if ( normflag || thickenflag ) calc_vertex_normal(fe,gdata[i].norm);
          fe = get_next_edge(fe);
        }

       if ( ridge_color_flag )
        {
          REAL side[FACET_EDGES][MAXCOORD];
          REAL otherside[FACET_EDGES][MAXCOORD];
         
          fe = get_facet_fe(f_id);
          for ( i = 0 ; i < FACET_EDGES ; i++ )
            { get_fe_side(fe,side[i]);
              get_fe_side(get_next_edge(get_next_facet(fe)),otherside[i]);
              fe = get_next_edge(fe);
            }
          for ( i = 0 ; i < FACET_EDGES ; i++ )
            if ( triple_prod(side[i],side[(i+1)%FACET_EDGES],otherside[i]) < 0.0 )
               gdata[i].color = RIDGE;
            else gdata[i].color = VALLEY; 
        }
      }

      /* call device-specific routine */
      if ( web.torus_flag && web.torus_clip_flag ) torus_clip(gdata,0);
      else (*graph_facet)(gdata,f_id);
    }
}

/*****************************************************************
*
*  function: plain_edges()
*
*  purpose:  plots all edges one by one.
*
*/

void plain_edges()
{
  int i,k;
  edge_id e_id;
  REAL *verts[2]; /* for adjusted triangle vertices */
  struct graphdata gdata[2];

  for ( i = 0 ; i < 2 ; i++ ) verts[i] = gdata[i].x;
  FOR_ALL_EDGES(e_id)
    { 
      if ( breakflag ) break;
      q_id = e_id;
      if ( show_expr.root && !eval(&show_expr,NULL) ) continue;

      /* get vertices */
      verts[0] = get_coord(get_edge_tailv(e_id));
      verts[1] = get_coord(get_edge_headv(e_id));
      for ( i = 0 ; i < web.sdim ; i++ )
        gdata[0].x[i] = verts[0][i];
      if ( web.symmetry_flag )
        (*sym_wrap)(verts[1],gdata[1].x,get_edge_wrap(e_id));
      else
        for ( i = 0 ; i < web.sdim ; i++ )
          gdata[1].x[i] = verts[1][i];
      gdata[0].color = get_edge_color(e_id);
      gdata[0].id = e_id;

      /* call device-specific routine */
      if ( web.modeltype == LINEAR )
       {
         if ( web.torus_flag && web.torus_clip_flag ) 
           torus_edge_clip(gdata,0);
         else (*graph_edge)(gdata);
       }
      else /* quadratic, so plot in 8 segments */
       { REAL *midx = get_coord(get_edge_midv(e_id));
	 REAL headx[MAXCOORD];
	 for ( i = 0 ; i < web.sdim ; i++ ) headx[i] = gdata[1].x[i];
	 for ( k = 1 ; k <= 8 ; k++ )
	   { for ( i = 0 ; i < web.sdim ; i++ )
	       gdata[1].x[i] = (1-k/8.)*(1-k/4.)*verts[0][i]
			      + k/2.*(1-k/8.)*midx[i]
			      - k/8.*(1-k/4.)*headx[i];
             if ( web.torus_flag && web.torus_clip_flag ) 
               torus_edge_clip(gdata,0);
             else (*graph_edge)(gdata);
	     for ( i = 0 ; i < web.sdim ; i++ ) gdata[0].x[i] = gdata[1].x[i];
	   }
       }
    }
}


/*****************************************************************
*
*  function: bare_edges()
*
*  purpose:  plots all facetless edges as skinny facets.
*
*/

void bare_edges()
{
  int i;
  edge_id e_id;
  REAL *verts[2]; /* for adjusted triangle vertices */
  struct graphdata gdata[3];
  int bare_count = 0;

  for ( i = 0 ; i < 2 ; i++ ) verts[i] = gdata[i].x;
  FOR_ALL_EDGES(e_id)
    { 
      if ( breakflag ) break;
      if ( valid_id(get_edge_fe(e_id)) ) continue;
      q_id = e_id;
      if ( show_expr.root && !eval(&show_expr,NULL) ) continue;
      bare_count++;
      if ( bare_count >= bare_edge_count ) continue; /* out of space */

      /* get vertices */
      verts[0] = get_coord(get_edge_tailv(e_id));
      verts[1] = get_coord(get_edge_headv(e_id));
      for ( i = 0 ; i < web.sdim ; i++ )
        gdata[0].x[i] = verts[0][i];
      if ( web.symmetry_flag )
        (*sym_wrap)(verts[1],gdata[1].x,get_edge_wrap(e_id));
      else for ( i = 0 ; i < web.sdim ; i++ )
        gdata[1].x[i] = verts[1][i];
      for ( i = 0 ; i < web.sdim ; i++ )
        gdata[2].x[i] = gdata[1].x[i];

      /* call device-specific routine */
      if ( web.torus_flag && web.torus_clip_flag ) torus_clip(gdata,0);
      else (*graph_facet)(gdata,NULLFACET);
    }
  bare_edge_count = bare_count+5;  /* for next time around */
}


/*****************************************************************
*
*  function: triple_edges()
*
*  purpose:  plots all multiple-facet edges if triple_show_flag set.
*
*/

void triple_edges()
{
  int i;
  edge_id e_id;
  facetedge_id fe_id;
  REAL *verts[2]; /* for adjusted triangle vertices */
  struct graphdata gdata[3];

  for ( i = 0 ; i < 2 ; i++ ) verts[i] = gdata[i].x;
  FOR_ALL_EDGES(e_id)
    { 
      if ( breakflag ) break;
      fe_id = get_edge_fe(e_id);
      if ( equal_id(get_next_facet(fe_id),get_prev_facet(fe_id)) ) continue;
      q_id = e_id;
      if ( show_expr.root && !eval(&show_expr,NULL) ) continue;

      /* get vertices */
      verts[0] = get_coord(get_edge_tailv(e_id));
      verts[1] = get_coord(get_edge_headv(e_id));
      for ( i = 0 ; i < web.sdim ; i++ )
        gdata[0].x[i] = verts[0][i];
      if ( web.symmetry_flag )
        (*sym_wrap)(verts[1],gdata[1].x,get_edge_wrap(e_id));
      else
        for ( i = 0 ; i < web.sdim ; i++ )
          gdata[1].x[i] = verts[1][i];
      gdata[0].id = e_id;

      /* call device-specific routine */
      if ( web.torus_flag && web.torus_clip_flag ) torus_edge_clip(gdata,0);
      else (*graph_edge)(gdata);
    }
}

/**********************************************************************
* 
*  Function: torus_edge_clip()
*
*  Purpose:  Recursive routine to subdivide edges crossing 
*            torus cell faces.  Edges have been already 
*            unwrapped by torus periods.
*/

void torus_edge_clip(gdata,m)
struct graphdata *gdata;  /* triangle to check */
int m;                    /* coordinate number */
{
  struct graphdata gdata1[2];  /* for possible subdivided edge */
  int i,k;
  REAL t,a,b;
  int wrap[2];

  /* see if any vertices outside cell in this coordinate */
  for ( i = 0 ; i < 2 ; i++ )
    wrap[i] = (int)floor(dot(web.inverse_periods[m],gdata[i].x,web.sdim));

  /* split, if necessary */
  if ( wrap[0] != wrap[1] )
    {
      int cut = (wrap[0] > wrap[1]) ? wrap[0] : wrap[1];

      /* set up head of new edge */
      gdata1[1] = gdata[1];
      gdata1[0].id = gdata[0].id;

      /* calculate new vertex */
      a = dot(web.inverse_periods[m],gdata[0].x,web.sdim);
      b = dot(web.inverse_periods[m],gdata1[1].x,web.sdim);
      t = (cut - a)/(b - a);
      for ( k = 0 ; k < web.sdim ; k++ )
        gdata1[0].x[k] = gdata[1].x[k] =
           (1 - t)*gdata[0].x[k] + t*gdata1[1].x[k];

      /* wrap new edge vertices properly */
      for ( i = 0 ; i < 2 ; i++ )
       for ( k = 0 ; k < web.sdim ; k++ )
          gdata1[i].x[k] -= wrap[1]*web.torus_period[m][k];
 
      /* send on for further check, or plot */
      gdata1[0].color = gdata[0].color;
      if ( m == web.sdim-1 ) 
        (*graph_edge)(gdata1);
      else
        torus_edge_clip(gdata1,m+1);
     }

   /* wrap vertices properly */
   for ( i = 0 ; i < 2 ; i++ )
     for ( k = 0 ; k < web.sdim ; k++ )
       gdata[i].x[k] -=  wrap[0]*web.torus_period[m][k];

   /* send on original edge structure */
   if ( m == web.sdim-1 ) 
     (*graph_edge)(gdata);
   else
     torus_edge_clip(gdata,m+1);

}

/**********************************************************************
* 
*  Function: torus_clip()
*
*  Purpose:  Recursive routine to subdivide triangles crossing 
*            torus cell faces 
*/

void torus_clip(gdata,m)
struct graphdata *gdata;  /* triangle to check */
int m;                    /* coordinate number */
{
  struct graphdata gdata1[FACET_VERTS],gdata2[FACET_VERTS];
  int i,k;
  REAL t,d,a,b;
  int wrap[FACET_VERTS];
  int cut,oddman,oddflag;
  int oddwrap;

  /* see if any vertices outside cell in this coordinate */
  for ( i = 0 ; i < FACET_VERTS ; i++ )
    wrap[i] = (int)floor(dot(web.inverse_periods[m],gdata[i].x,web.sdim));

  /* find odd vertex */
  oddflag = 1;
  if ( (wrap[0] == wrap[1]) && (wrap[1] == wrap[2]) )
    { oddflag = 0;  oddman = 0; }
  else if ( wrap[0] == wrap[1] ) oddman = 2;
  else if ( wrap[1] == wrap[2] ) oddman = 0;
  else if ( wrap[2] == wrap[0] ) oddman = 1;
  else
   { fprintf(stderr,"Triangle spans too many periods. Wraps %d %d %d\n",
         wrap[0],wrap[1],wrap[2]);
     oddman = 0;
   }

  /* wrap new triangle vertices properly */
  oddwrap = wrap[oddman];
  for ( i = 0 ; i < FACET_VERTS ; i++ )
   { 
     wrap[i] -= oddwrap;
     for ( k = 0 ; k < web.sdim ; k++ )
      { d = oddwrap*web.torus_period[m][k];
        gdata[i].x[k] -= d;
      }
   }


  /* split, if necessary */
  if ( oddflag )
    {
      int pair1 = (oddman+1)%FACET_VERTS;
      int pair2 = (oddman+2)%FACET_EDGES;

      /* find wrap multiple that cuts triangle */
      if ( wrap[oddman] < wrap[pair1] )
        { cut = wrap[pair1];
          oddflag = 1;
        }
      else
        { cut = wrap[oddman];
          oddflag = -1;
        }

      /* set up new triangles */
      for ( i = 0 ; i < FACET_VERTS ; i++ )
        { gdata2[i] = gdata[i];
          gdata1[i] = gdata[i];
        } 

      /* calculate new vertices */
      a = dot(web.inverse_periods[m],gdata[oddman].x,web.sdim);
      b = dot(web.inverse_periods[m],gdata1[pair1].x,web.sdim);
      if ( fabs(a-b) < 0.00000000001 ) t = 0.5;
      else t = (cut - a)/(b - a);
      for ( k = 0 ; k < web.sdim ; k++ )
        gdata1[oddman].x[k] = gdata[pair1].x[k] =
           (1 - t)*gdata[oddman].x[k] + t*gdata1[pair1].x[k];
      b = dot(web.inverse_periods[m],gdata2[pair2].x,web.sdim);
      if ( fabs(a-b) < 0.00000000001 ) t = 0.5;
      else t = (cut - a)/(b - a);
      for ( k = 0 ; k < web.sdim ; k++ )
        gdata2[oddman].x[k] = gdata[pair2].x[k] = gdata1[pair2].x[k] =
           (1 - t)*gdata[oddman].x[k] + t*gdata2[pair2].x[k];

      /* wrap new triangle vertices properly */
      for ( i = 0 ; i < FACET_VERTS ; i++ )
       for ( k = 0 ; k < web.sdim ; k++ )
        { d = wrap[pair1]*web.torus_period[m][k];
          gdata1[i].x[k] -= d;
          gdata2[i].x[k] -= d;
        }
 
      /* send on for further check, or plot */
      if ( m == web.sdim-1 ) 
        { (*graph_facet)(gdata1,NULLFACET);
          (*graph_facet)(gdata2,NULLFACET);
        }
      else
        { torus_clip(gdata1,m+1);
          torus_clip(gdata2,m+1);
        }
     }

   /* send on original triangle structure */
   if ( m == web.sdim-1 ) 
     (*graph_facet)(gdata,NULLFACET);
   else
     torus_clip(gdata,m+1);

}

/* comparison for ordering structures */
int bfcomp(a,b)
struct bodyface *a,*b;
{
  if ( ordinal(a->f_id) < ordinal(b->f_id) ) return -1;
  if ( ordinal(a->f_id) > ordinal(b->f_id) ) return 1;
  if ( inverted(a->f_id) < inverted(b->f_id) ) return -1;
  if ( inverted(a->f_id) > inverted(b->f_id) ) return 1;

  return 0;
}

/**********************************************************************
*
*  Function: torus_bodies()
*
*  Purpose:  To display SOAPFILM torus model in terms of connected
*            bodies.  Finds all facets of each body and plots them
*            in a connected manner.
*/

void torus_bodies()
{
  struct bodyface *faces;
  int facemax;
  facetedge_id fe,fen,ff,fa,fan;
  int facetop;
  body_id b_id;
  int i,j,k;
  int numleft;
  struct bodyface *bf;
  struct graphdata gdata[FACET_VERTS];
  facet_id f_id;
  WRAPTYPE xwrap; /* for centering body */

  facemax = 2*web.skel[FACET].count;
  faces = (struct bodyface *) temp_calloc(facemax,sizeof(struct bodyface));

  FOR_ALL_BODIES(b_id)
   {
      if ( breakflag ) break;
      q_id = b_id;
      if ( show_expr.root && !eval(&show_expr,NULL) ) continue;

     facetop = 0;

     /* first, get a list of all facets bounding the body,
        with proper orientation. Some may be included twice
        with opposite orientations in case the body wraps
        around the torus.
      */
     FOR_ALL_FACETS(f_id)
       {
         if ( equal_id(b_id,get_facet_body(f_id)) )
           {
             faces[facetop].wrapflag = 0;
             faces[facetop++].f_id = f_id;      
           }
         if ( equal_id(b_id,get_facet_body(facet_inverse(f_id))) )
           {
             faces[facetop].wrapflag = 0;
             faces[facetop++].f_id = facet_inverse(f_id);       
           }
       }

     if ( facetop == 0 ) return;

     /* sort in facet order, so can find facets quickly */
     qsort((char *)faces,facetop,sizeof(struct bodyface),FCAST bfcomp);

     /* Now go through list repeatedly, marking proper wraps
        for facet base vertices as can.
      */
     faces[0].wrap = 0; /* wrap to base cell */
     faces[0].wrapflag = 1;

     /* go through rest of list */
     numleft = facetop-1;
     do
       {
         for ( k = 1 ; k < facetop ; k++ )
          {
            if ( faces[k].wrapflag ) continue;
            fe = get_facet_fe( faces[k].f_id );
            for ( i = 0 ; i < FACET_VERTS ; i++ )  /* check for wrapped neighbor */
              {
                fen = fe_inverse(get_prev_facet(fe));
                ff = get_fe_facet(fen);
                bf = (struct bodyface *)bsearch((char *)&ff,(char *)faces,
                    facetop, sizeof(struct bodyface),FCAST bfcomp);
                if ( bf == NULL )
                  { error("torus_bodies: missing face.\n",WARNING);
                    continue;
                  }
                if ( bf-> wrapflag == 0 ) 
                  { fe = get_next_edge(fe);
                    continue;
                  }
           
                /* now have wrapped neighbor */ 

                /* start at base point of neighbor and follow edges
                   accumulating wraps until new base point is reached */

                faces[k].wrap = bf->wrap; 
                fan = get_facet_fe(bf->f_id);
                fen = get_next_edge(fen);  /* so tail of fan winds up at
                                              tail of fe */
                while ( !equal_id(fan,fen) )
                  { /* walk around neighbor */
                      faces[k].wrap =
			(*sym_compose)(faces[k].wrap,get_fe_wrap(fan)); 
                    fan = get_next_edge(fan);
                  }
                fa = get_facet_fe(faces[k].f_id);
                while ( !equal_id(fa,fe) )
                  { /* walk backward around new facet */
                    fe = get_prev_edge(fe);
	            faces[k].wrap = (*sym_compose)(faces[k].wrap,
		       		       (*sym_inverse)(get_fe_wrap(fe))); 
                  }
               
                faces[k].wrapflag = 1;
                numleft--;
                break; 
              } 
          }
      } while ( numleft > 0 );

    /* try to center body in cell by finding most common wrap */
    { struct { WRAPTYPE wrap; int count; } xxwrap[50];
      int wrapcount = 1;  /* number of different wraps */
       memset((char*)xxwrap,0,sizeof(xxwrap));
       for ( k = 0 ; k < facetop ; k++ )
	 { for ( i = 0 ; i < wrapcount ; i++ )
	     if ( xxwrap[i].wrap == faces[k].wrap )
	       xxwrap[i].count++;
	   if ( (i == wrapcount) && (i < 50-1) )
	     { xxwrap[wrapcount].wrap = faces[k].wrap;
	       xxwrap[wrapcount++].count = 1;
	     }
	 }
       for ( k = 0, i = 0 ; i < wrapcount ; i++ )
	 if ( xxwrap[i].count > xxwrap[k].count ) k = i;
       xwrap = (*sym_inverse)(xxwrap[k].wrap);
     }
    
    /* now plot all the facets */
    for ( k = 0 ; k < facetop ; k++ )
      { WRAPTYPE wrap;
        if ( breakflag ) break;

        f_id = faces[k].f_id;
        if ( get_fattr(f_id) & NODISPLAY ) continue;
        if ( get_facet_color(f_id) == CLEAR ) continue;
	fe = get_facet_fe(f_id);
	wrap = (*sym_compose)(faces[k].wrap,xwrap);
        for ( i = 0 ; i < FACET_VERTS ; i++ )   /* vertex loop */
	 { double *verts;
	   verts = get_coord(get_fe_tailv(fe));
	   (*sym_wrap)(verts,gdata[i].x,wrap);
	   gdata[i].x[HOMDIM-1] = 1.0;  /* homogeneous coord */
           if ( colorflag )  gdata[i].color = ordinal(b_id);
	   wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
	   fe = get_next_edge(fe);
         }


      /* do iner clipping, if called for */
      if ( inner_clip_flag )
        {
          for ( i = 0 ; i < FACET_VERTS ; i++ )
           { double dist = 0.0;
             REAL *x = get_coord(web.zoom_v);

             for ( j = 0 ; j < web.sdim ; j++ )
               dist += (x[j]-gdata[i].x[j])*(x[j]-gdata[i].x[j]);
   
             if ( sqrt(dist) > inner_clip_rad ) break; /* it's a keeper */
           }
          if ( i == FACET_VERTS ) continue; /* entirely inside */
        }

      fe = get_facet_fe(f_id);
        if ( normflag || thickenflag  )
          {
            fe = get_facet_fe(f_id);
            for ( i = 0 ; i < FACET_VERTS ; i++ )
             { 
               calc_vertex_normal(fe,gdata[i].norm);
               fe = get_next_edge(fe);
             }
          }
  
        /* call device-specific routine */
        (*graph_facet)(gdata,f_id);
       }
     
    }  /* end bodies */

   temp_free((char *)faces);
}


/**********************************************************************
*
*  Function: torus_cells()
*
*  Purpose:  To display STRING torus model in terms of connected
*            cells. Assumes everything in x-y plane.  Graphs edges
*            around each facet in a connected manner.
*/

void torus_cells()
{
  facet_id f_id;
  struct graphdata gdata[2];
  REAL side[MAXCOORD];
  REAL *x;
  facetedge_id fe_id,fe;
  int i; 

  FOR_ALL_FACETS(f_id)
   {
     q_id = f_id;
     if ( show_expr.root && !eval(&show_expr,NULL) ) continue;
     if ( breakflag ) break;
     fe_id = get_facet_fe(f_id);
     x = get_coord(get_fe_tailv(fe_id));
     for ( i = 0 ; i < web.sdim ; i++ )
       gdata[0].x[i] = x[i];
     fe = fe_id;
     do
       {
         if ( !valid_id(fe) ) break;
         get_fe_side(fe,side);
         for ( i = 0 ; i < web.sdim ; i++ )
           gdata[1].x[i] = gdata[0].x[i] + side[i];
         gdata[0].id = get_fe_edge(fe);
         (*graph_edge)(gdata);
         gdata[0] = gdata[1];
         fe = get_next_edge(fe);
       }
     while ( !equal_id(fe,fe_id) );
   }
}

