#include <stdio.h>
#include <varargs.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#ifdef IRIX
#include <sys/termio.h>
#include <sys/fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <utmp.h>
#else
#include <sys/file.h>
#include <sgtty.h>
#endif IRIX

#include "lxt.h"

#include "xvcursor.bitmap"
#include "xvvscursor.bitmap"
#include "xvhscursor.bitmap"

extern Pixmap xvcursor_pm, xvvscursor_pm, xvhscursor_pm;
extern boolean xvcursor_init;
extern boolean xp_fullredraw;
extern boolean xp_usrresize;
extern Void *lxt_selsrc;
extern Void *lxt_seldest;
extern int lxt_seldesttype;

#include "grey.bitmap"

Panel *xp_panels= (Panel *) NULL;
static boolean xp_init= FALSE;

/* special keys for text items */
char xpkey_null, xpkey_tab, xpkey_cret, xpkey_backspace;
char xpkey_wordkill, xpkey_linekill;

/* return value for panel{item}_get() */
Void *xpret_void;
int xpret_int;
boolean xpret_bool;

extern void xpi_std_dsp_proc(), xpi_null_proc();
extern int xplabel_sz_proc(), xplabel_idef_proc(), xplabel_dst_proc();
extern void xplabel_drw_proc();
extern int xptext_sz_proc(), xptext_idef_proc(), xptext_dst_proc();
extern void xptext_drw_proc();
extern void xptext_bp_proc(), xptext_br_proc(), xptext_pm_proc();
extern void xptext_kp_proc();
extern int xpbutton_sz_proc(), xpbutton_idef_proc(), xpbutton_dst_proc();
extern void xpbutton_drw_proc();
extern void xpbutton_bp_proc(), xpbutton_br_proc(), xpbutton_pm_proc();
extern int xpslider_sz_proc(), xpslider_idef_proc(), xpslider_dst_proc();;
extern void xpslider_drw_proc();
extern void xpslider_bp_proc(), xpslider_br_proc(), xpslider_pm_proc();
extern int xpenum_sz_proc(), xpenum_idef_proc(), xpenum_dst_proc();
extern void xpenum_drw_proc();
extern void xpenum_bp_proc();

/* dispatch table for all item resize, draw,
   display and event handling routines */
Panelitem_dispatch panelitem_proctab[]= {
	{ xplabel_idef_proc, xplabel_dst_proc,
	  xplabel_sz_proc, xplabel_drw_proc, xpi_std_dsp_proc,
	  xpi_null_proc, xpi_null_proc, xpi_null_proc, xpi_null_proc },
	{ xptext_idef_proc, xptext_dst_proc,
	  xptext_sz_proc, xptext_drw_proc, xpi_std_dsp_proc,
	  xptext_bp_proc, xptext_br_proc, xptext_pm_proc, xptext_kp_proc },
	{ xpbutton_idef_proc, xpbutton_dst_proc,
	  xpbutton_sz_proc, xpbutton_drw_proc, xpi_std_dsp_proc,
	  xpbutton_bp_proc, xpbutton_br_proc, xpbutton_pm_proc, xpi_null_proc },
	{ xpslider_idef_proc, xpslider_dst_proc,
	  xpslider_sz_proc, xpslider_drw_proc, xpi_std_dsp_proc,
	  xpslider_bp_proc, xpslider_br_proc, xpslider_pm_proc, xpi_null_proc },
	{ xpenum_idef_proc, xpenum_dst_proc,
	  xpenum_sz_proc, xpenum_drw_proc, xpi_std_dsp_proc,
	  xpenum_bp_proc, xpi_null_proc, xpi_null_proc, xpi_null_proc },
	{ xpenum_idef_proc, xpenum_dst_proc,
	  xpenum_sz_proc, xpenum_drw_proc, xpi_std_dsp_proc,
	  xpenum_bp_proc, xpi_null_proc, xpi_null_proc, xpi_null_proc },
	{ xpenum_idef_proc, xpenum_dst_proc,
	  xpenum_sz_proc, xpenum_drw_proc, xpi_std_dsp_proc,
	  xpenum_bp_proc, xpi_null_proc, xpi_null_proc, xpi_null_proc },
	0
};

/*VARARGS*/
Panel *
panel_create(va_alist)
/*
   User-callable.
   Creates an empty panel.
*/
va_dcl
{
	va_list varg_ptr;
	int nargs;
	Panel *p;
	Display *dpy;
	Window win;
	int attr;
	char *name;
	GC gc;
	XGCValues gcv;
	XSetWindowAttributes xswa;
	XWindowAttributes xwa;
	XImage image;
	XColor fg_xc, bg_xc;
	Colormap cmap;
	int panel_filldefaults();
	int panel_config();
	void panel_init();
	void panel_draw(), panel_display();
	void panelvscroll_draw(), panelhscroll_draw();

	p= (Panel *) NULL;
	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get program name */
		if (nargs == 0) {
			name= va_arg(varg_ptr, char *);
			if (name == (char *) NULL) {
				(void) fprintf(stderr, "panel_create: null program name\n");
				return((Panel *) NULL);
			}
		}

		/* get display */
		else if (nargs == 1) {
			dpy= va_arg(varg_ptr, Display *);
			if (dpy == (Display *) NULL) {
				(void) fprintf(stderr, "panel_create: null display\n");
				return((Panel *) NULL);
			}
		}

		/* get enclosing window */
		else if (nargs == 2) {
			win= va_arg(varg_ptr, Window);
			if (win == None) {
				(void) fprintf(stderr, "panel_create: null parent window\n");
				return((Panel *) NULL);
			}
			if ((p= (Panel *) calloc(1, sizeof(Panel))) == (Panel *) NULL) {
				(void) fprintf(stderr, "panel_create: memory allocation error\n");
				return((Panel *) NULL);
			}
			p->xp_magic= LX_PANEL;
			p->xp_dpy= dpy;
			p->xp_win= win;
			if (panel_filldefaults(name, p) != LX_SUCCESS) {
				cfree((char *) p);
				return((Panel *) NULL);
			}
			if (!xp_init)
				panel_init(p->xp_dpy);
		}

		else {
			char *c;
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXP_NULL)
				break;

			switch (attr) {
			case LXP_FONT:
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL) {
					(void) fprintf(stderr, "panel_create: null fontname\n");
					cfree((char *) p);
					return((Panel *) NULL);
				}
				if ((p->xp_font= XLoadQueryFont(p->xp_dpy, c)) == NULL) {
					(void) fprintf(stderr, "panel_create: cannot load font %s\n", c);
					cfree((char *) p);
					return((Panel *) NULL);
				}
				cfree(p->xp_fontnm);
				if ((p->xp_fontnm= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
					(void) fprintf(stderr, "panel_create: memory allocation error\n");
					cfree((char *) p);
					return((Panel *) NULL);
				}
				(void) strcpy(p->xp_fontnm, c);
				break;
			case LXP_FOREGROUND:
				p->xp_fg= va_arg(varg_ptr, unsigned long);
				break;
			case LXP_BACKGROUND:
				p->xp_bg= va_arg(varg_ptr, unsigned long);
				break;
			case LXP_WIDTH:
				p->xp_rpw= va_arg(varg_ptr, int);
				p->xp_flags|= LXP_REQWIDTH;
				break;
			case LXP_HEIGHT:
				p->xp_rph= va_arg(varg_ptr, int);
				p->xp_flags|= LXP_REQHEIGHT;
				break;
			case LXP_BOTTOMMARGIN:
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panel_create: illegal margin size\n");
				else
					p->xp_bmargin= n;
				break;
			case LXP_RIGHTMARGIN:
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panel_create: illegal margin size\n");
				else
					p->xp_rmargin= n;
				break;
			case LXP_BARWIDTH:
				p->xp_vscroll->xs_barwidth= p->xp_hscroll->xs_barwidth= va_arg(varg_ptr, int);
				break;
			case LXP_BUBBLEMARGIN:
				p->xp_vscroll->xs_bubblemargin= p->xp_hscroll->xs_bubblemargin= va_arg(varg_ptr, int);
				break;
			case LXP_BUTTONLEN:
				p->xp_vscroll->xs_buttonlen= p->xp_hscroll->xs_buttonlen= va_arg(varg_ptr, int);
				break;
			case LXP_PCURSOR:
				p->xp_pcursor= va_arg(varg_ptr, Cursor);
				break;
			case LXP_VSCURSOR:
				p->xp_vscursor= va_arg(varg_ptr, Cursor);
				break;
			case LXP_HSCURSOR:
				p->xp_hscursor= va_arg(varg_ptr, Cursor);
				break;
			case LXP_VISIBLE:
				n= va_arg(varg_ptr, int);
				if (n)
					p->xp_flags|= LXP_PANELVISIBLE;
				else
					p->xp_flags&= ~LXP_PANELVISIBLE;
				break;
			case LXP_CLIENTDATA:
				p->xp_clientdata= va_arg(varg_ptr, char *);
				break;
			case LXP_PROC:
				p->xp_proc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXP_GRAB:
				n= va_arg(varg_ptr, int);
				if (n)
					p->xp_flags|= LXP_GRABPTR;
				else
					p->xp_flags&= ~LXP_GRABPTR;
				break;
			case LXP_ACTIVE:
				n= va_arg(varg_ptr, int);
				if (n)
					p->xp_flags|= LXP_PANELACTIVE;
				else
					p->xp_flags&= ~LXP_PANELACTIVE;
				break;
			case LXP_ALARMINTERVAL:
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panel_create: illegal alarm interval\n");
				else
					p->xp_alarmint= n;
				break;
			case LXP_ALARMPROC:
				p->xp_alarmproc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXP_VWIDTH:
			case LXP_VHEIGHT:
			case LXP_VSCROLL:
			case LXP_HSCROLL:
			case LXP_DWIDTH:
			case LXP_DHEIGHT:
				(void) fprintf(stderr, "panel_create: attribute is query-only\n");
				break;
			default:
				(void) fprintf(stderr, "panel_create: ignoring unrecognized attribute\n");
				break;
			}
		}
	}
	va_end(varg_ptr);

	XGetWindowAttributes(p->xp_dpy, p->xp_win, &xwa);
	p->xp_pwin= XCreateSimpleWindow(p->xp_dpy, p->xp_win, 0, 0, xwa.width, xwa.height, 0, p->xp_fg, p->xp_bg);
	p->xp_vswin= XCreateSimpleWindow(p->xp_dpy, p->xp_win, 0, 0, xwa.width, xwa.height, 0, p->xp_fg, p->xp_bg);
	p->xp_hswin= XCreateSimpleWindow(p->xp_dpy, p->xp_win, 0, 0, xwa.width, xwa.height, 0, p->xp_fg, p->xp_bg);

	xswa.colormap= xwa.colormap;
	XChangeWindowAttributes(p->xp_dpy, p->xp_pwin, CWColormap, &xswa);
	XChangeWindowAttributes(p->xp_dpy, p->xp_vswin, CWColormap, &xswa);
	XChangeWindowAttributes(p->xp_dpy, p->xp_hswin, CWColormap, &xswa);

	XSelectInput(p->xp_dpy, p->xp_win, ExposureMask | StructureNotifyMask);
	if (p->xp_flags & LXP_PANELACTIVE) {
		XSelectInput(p->xp_dpy, p->xp_pwin, (ExposureMask | ButtonPressMask | ButtonReleaseMask | ButtonMotionMask | KeyPressMask | PropertyChangeMask));
		XSelectInput(p->xp_dpy, p->xp_vswin, (ExposureMask | ButtonPressMask));
		XSelectInput(p->xp_dpy, p->xp_hswin, (ExposureMask | ButtonPressMask));
	}
	else {
		XSelectInput(p->xp_dpy, p->xp_pwin, ExposureMask | PropertyChangeMask);
		XSelectInput(p->xp_dpy, p->xp_vswin, ExposureMask);
		XSelectInput(p->xp_dpy, p->xp_hswin, ExposureMask);
	}

	/* create grey stipple */
	p->xp_stipplepm= XCreatePixmap(p->xp_dpy, DefaultRootWindow(dpy), grey_width, grey_height, 1);
	gcv.foreground= p->xp_fg;
	gcv.background= p->xp_bg;
	gc= XCreateGC(dpy, p->xp_stipplepm, GCForeground | GCBackground, &gcv);
	image.height= grey_height;
	image.width= grey_width;
	image.xoffset= 0;
	image.format= XYBitmap;
	image.data= grey_bits;
	image.byte_order= LSBFirst;
	image.bitmap_unit= 8;
	image.bitmap_bit_order= LSBFirst;
	image.bitmap_pad= 8;
	image.bytes_per_line= (grey_width+7)>>3;
	image.depth= 1;
	XPutImage(dpy, p->xp_stipplepm, gc, &image, 0, 0, 0, 0, grey_width, grey_height);
	XFreeGC(dpy, gc);

	/* create cursors */
	cmap= xwa.colormap;
	fg_xc.pixel= p->xp_fg;
	XQueryColor(p->xp_dpy, cmap, &fg_xc);
	fg_xc.flags= DoRed | DoGreen | DoBlue;
	bg_xc.pixel= p->xp_bg;
	XQueryColor(p->xp_dpy, cmap, &bg_xc);
	bg_xc.flags= DoRed | DoGreen | DoBlue;
	if (p->xp_pcursor == None)
		p->xp_pcursor= XCreatePixmapCursor(p->xp_dpy, xvcursor_pm, xvcursor_pm, &fg_xc, &bg_xc, xvcursor_x_hot, xvcursor_y_hot);
	if (p->xp_vscursor == None)
		p->xp_vscursor= XCreatePixmapCursor(p->xp_dpy, xvvscursor_pm, xvvscursor_pm, &fg_xc, &bg_xc, xvvscursor_x_hot, xvvscursor_y_hot);
	if (p->xp_hscursor == None)
		p->xp_hscursor= XCreatePixmapCursor(p->xp_dpy, xvhscursor_pm, xvhscursor_pm, &fg_xc, &bg_xc, xvhscursor_x_hot, xvhscursor_y_hot);
	XDefineCursor(p->xp_dpy, p->xp_pwin, p->xp_pcursor);
	XDefineCursor(p->xp_dpy, p->xp_vswin, p->xp_vscursor);
	XDefineCursor(p->xp_dpy, p->xp_hswin, p->xp_hscursor);

	/* initialize GCs */
	gcv.font= p->xp_font->fid;
	gcv.foreground= p->xp_fg;
	gcv.background= p->xp_bg;
	gcv.plane_mask= AllPlanes;
	gcv.function= GXcopy;
	p->xp_gc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);
	gcv.plane_mask= p->xp_fg ^ p->xp_bg;
	gcv.function= GXinvert;
	p->xp_igc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCFunction | GCPlaneMask), &gcv);
	gcv.function= GXcopy;
	gcv.stipple= p->xp_stipplepm;
	gcv.fill_style= FillStippled;
	p->xp_sgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);
	gcv.foreground= p->xp_bg;
        gcv.background= p->xp_fg;
	gcv.plane_mask= AllPlanes;
        gcv.function= GXcopy;
        p->xp_cgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);

	p->xp_next= xp_panels;
	xp_panels= p;

	switch (xwa.map_state) {
	case IsUnviewable:
	case IsViewable:
		p->xp_flags|= LXP_FRAMEMAPPED;
		(void) panel_config(p);
		p->xp_flags|= LXP_CONFIGDONE;
		XMapWindow(p->xp_dpy, p->xp_pwin);
		panel_draw(p);
		panel_display(p);
		if (p->xp_flags & LXP_VSCROLLVIS) {
			XMapWindow(p->xp_dpy, p->xp_vswin);
			panelvscroll_draw(p);
		}
		if (p->xp_flags & LXP_HSCROLLVIS) {
			XMapWindow(p->xp_dpy, p->xp_hswin);
			panelhscroll_draw(p);
		}
		break;
	case IsUnmapped:
		p->xp_flags&= ~LXP_FRAMEMAPPED;
		break;
	default:
		break;
	}
	return(p);
}

int
panel_filldefaults(name, p)
char *name;
Panel *p;
{
	char *xdef, fontname[LX_MAXFONTNMLEN+1];
	int screen;

	p->xp_vx= p->xp_vy= 0;
	if ((p->xp_name= calloc((unsigned) (strlen(name)+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "panel_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	(void) strcpy(p->xp_name, name);

	/* font */
	if ((xdef= XGetDefault(p->xp_dpy, p->xp_name, "Font")) == (char *) NULL)
		(void) strcpy(fontname, LXPDEF_FONT);
	else
		(void) strncpy(fontname, xdef, LX_MAXFONTNMLEN);
	if ((p->xp_font= XLoadQueryFont(p->xp_dpy, fontname)) == NULL) {
		(void) fprintf(stderr, "panel_filldefaults: cannot load font %s\n", fontname);
		return(LX_ERROR);
	}
	if ((p->xp_fontnm= calloc((unsigned) (strlen(fontname)+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "panel_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	(void) strcpy(p->xp_fontnm, fontname);

	/* internal border width */
	if ((xdef= XGetDefault(p->xp_dpy, p->xp_name, "LXTInternalBorderWidth")) == (char *) NULL)
		p->xp_ibw= 0;
	else {
		p->xp_ibw= atoi(xdef);
		if (p->xp_ibw < 0)
			p->xp_ibw= 0;
	}

	/* foreground and background */
	screen= DefaultScreen(p->xp_dpy);
	p->xp_fg= BlackPixel(p->xp_dpy, screen);
	p->xp_bg= WhitePixel(p->xp_dpy, screen);

	/* panel margins */
	p->xp_bmargin= LXPDEF_BOTTOMMARGIN;
	p->xp_rmargin= LXPDEF_RIGHTMARGIN;

	/* blocking alarm stuff */
	p->xp_alarmint= LXPDEF_ALARMINTERVAL;
	p->xp_alarmproc= (void (*)()) NULL;

	/* scrollbars */
	if ((p->xp_vscroll= (Scrollbar *) calloc(1, sizeof(Scrollbar))) == (Scrollbar *) NULL) {
		(void) fprintf(stderr, "panel_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	p->xp_vscroll->xs_type= LXS_VERTICAL;
	p->xp_vscroll->xs_barwidth= LXSDEF_BARWIDTH;
	p->xp_vscroll->xs_bubblemargin= LXSDEF_BUBBLEMARGIN;
	p->xp_vscroll->xs_buttonlen= LXSDEF_BUTTONLEN;
	if ((p->xp_hscroll= (Scrollbar *) calloc(1, sizeof(Scrollbar))) == (Scrollbar *) NULL) {
		(void) fprintf(stderr, "panel_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	p->xp_hscroll->xs_type= LXS_HORIZONTAL;
	p->xp_hscroll->xs_barwidth= LXSDEF_BARWIDTH;
	p->xp_hscroll->xs_bubblemargin= LXSDEF_BUBBLEMARGIN;
	p->xp_hscroll->xs_buttonlen= LXSDEF_BUTTONLEN;

	/* flags */
	p->xp_flags= LXP_PANELVISIBLE | LXP_GRABPTR | LXP_PANELACTIVE;

	/* null fields */
	p->xp_ppm= p->xp_vspm= p->xp_hspm= None;
	p->xp_pcursor= p->xp_vscursor= p->xp_hscursor= None;
	p->xp_items= (Panel_item *) NULL;
	p->xp_selitem= p->xp_seltext= (Panel_item *) NULL;
	p->xp_proc= (void (*)()) NULL;
	p->xp_next= (Panel *) NULL;
	return(LX_SUCCESS);
}

void
panel_init(dpy)
Display *dpy;
{
	int fd, ld;
#ifdef IRIX
	struct termio tty_ltchars;
	struct termio tty_params;
#else
	struct sgttyb tty_params;
	struct ltchars tty_ltchars;
#endif IRIX
	void cursor_init();

	/* create cursor pixmaps */
	if (!xvcursor_init)
		cursor_init(dpy);

	/* initialize default keyboard codes */
	xpkey_null= '\0';
	xpkey_backspace= '\177';
	xpkey_tab= '\011';
	xpkey_cret= '\015';
	xpkey_wordkill= '\027';
	xpkey_linekill= '\025';

	/* attempt to determine user's terminal characteristics */
#ifdef IRIX
        /* get tty characteristics from /dev/tty or stdin */
        if ((fd= open("/dev/tty", O_NDELAY)) < 0)
                fd= 0;
        if (ioctl(fd, TCGETA, &tty_params) < 0)
                (void) fprintf(stderr, "panel_init: ioctl error\n");
        else {
                xpkey_backspace= tty_params.c_cc[2];
                xpkey_linekill= tty_params.c_cc[3];
                xpkey_wordkill= tty_ltchars.c_cc[9];
        }
	if (fd >= 1)
		(void) close(fd);
#else
	/* get tty characteristics from /dev/tty or stdin */
	if ((fd= open("/dev/tty", O_RDONLY)) < 0)
		fd= 0;
	if (ioctl(fd, TIOCGETP, &tty_params) < 0)
		(void) fprintf(stderr, "panel_init: ioctl error\n");
	else {
		xpkey_backspace= tty_params.sg_erase;
		xpkey_linekill= tty_params.sg_kill;
	}

	/* new line discipline? */
	if (ioctl(fd, TIOCGETD, &ld) == 0) {
		if (ld == NTTYDISC) {
			if (ioctl(fd, TIOCGLTC, &tty_ltchars) < 0)
				(void) fprintf(stderr, "panel_init: ioctl error\n");
			else
				xpkey_wordkill= tty_ltchars.t_werasc;
		}
	}
	else
		(void) fprintf(stderr, "panel_init: ioctl error\n");
	if (fd >= 1)
		(void) close(fd);
#endif IRIX

	xp_fullredraw= FALSE;
	xp_usrresize= FALSE;
	xp_init= TRUE;
}

/*VARARGS*/
int
panel_set(va_alist)
/*
   User-callable.
   Changes an attribute of a previously created panel.
*/
va_dcl
{
	va_list varg_ptr;
	int nargs, attr;
	boolean redisplay, reconfig, undisplay;
	boolean visible;
	Panel *p;
	unsigned long plane_mask;
	void panel_unmapsubwins();

	redisplay= reconfig= undisplay= FALSE;

	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		if (nargs == 0) {
			p= va_arg(varg_ptr, Panel *);
			if (p == (Panel *) NULL) {
				(void) fprintf(stderr, "panel_set: null panel\n");
				return(LX_ERROR);
			}
			if (p->xp_magic != LX_PANEL) {
				(void) fprintf(stderr, "panel_set: object is not a panel\n");
				return(LX_ERROR);
			}
			if ((p->xp_flags & LXP_PANELVISIBLE) && (p->xp_flags & LXP_FRAMEMAPPED))
				visible= TRUE;
			else
				visible= FALSE;
		}

		else {
			char *c;
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXP_NULL)
				break;

			switch (attr) {
			case LXP_FONT:
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL) {
					(void) fprintf(stderr, "panel_create: null fontname\n");
					cfree((char *) p);
					return(LX_ERROR);
				}
				if ((p->xp_font= XLoadQueryFont(p->xp_dpy, c)) == NULL) {
					(void) fprintf(stderr, "panel_set: cannot load font %s\n", c);
					return(LX_ERROR);
				}
				cfree(p->xp_fontnm);
				if ((p->xp_fontnm= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
					(void) fprintf(stderr, "panel_set: memory allocation error\n");
					return(LX_ERROR);
				}
				(void) strcpy(p->xp_fontnm, c);
				XSetFont(p->xp_dpy, p->xp_gc, p->xp_font->fid);
				XSetFont(p->xp_dpy, p->xp_igc, p->xp_font->fid);
				XSetFont(p->xp_dpy, p->xp_cgc, p->xp_font->fid);
				XSetFont(p->xp_dpy, p->xp_sgc, p->xp_font->fid);
				if (visible)
					redisplay= TRUE;
				break;
			case LXP_FOREGROUND:
				p->xp_fg= va_arg(varg_ptr, unsigned long);
				XSetForeground(p->xp_dpy, p->xp_gc, p->xp_fg);
				XSetForeground(p->xp_dpy, p->xp_igc, p->xp_fg);
				XSetBackground(p->xp_dpy, p->xp_cgc, p->xp_fg);
				XSetForeground(p->xp_dpy, p->xp_sgc, p->xp_fg);
				plane_mask= p->xp_fg ^ p->xp_bg;
				XSetPlaneMask(p->xp_dpy, p->xp_igc, plane_mask);
				if (visible)
					redisplay= TRUE;
				break;
			case LXP_BACKGROUND:
				p->xp_bg= va_arg(varg_ptr, unsigned long);
				XSetBackground(p->xp_dpy, p->xp_gc, p->xp_bg);
				XSetBackground(p->xp_dpy, p->xp_igc, p->xp_bg);
				XSetForeground(p->xp_dpy, p->xp_cgc, p->xp_bg);
				XSetBackground(p->xp_dpy, p->xp_sgc, p->xp_bg);
				plane_mask= p->xp_fg ^ p->xp_bg;
				XSetPlaneMask(p->xp_dpy, p->xp_igc, plane_mask);
				if (visible)
					redisplay= TRUE;
				break;
			case LXP_WIDTH:
				p->xp_rpw= va_arg(varg_ptr, int);
				p->xp_flags|= LXP_REQWIDTH;
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXP_HEIGHT:
				p->xp_rph= va_arg(varg_ptr, int);
				p->xp_flags|= LXP_REQHEIGHT;
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXP_BARWIDTH:
				p->xp_vscroll->xs_barwidth= p->xp_hscroll->xs_barwidth= va_arg(varg_ptr, int);
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXP_BUBBLEMARGIN:
				p->xp_vscroll->xs_bubblemargin= p->xp_hscroll->xs_bubblemargin= va_arg(varg_ptr, int);
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXP_BUTTONLEN:
				p->xp_vscroll->xs_buttonlen= p->xp_hscroll->xs_buttonlen= va_arg(varg_ptr, int);
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXP_RIGHTMARGIN:
				n= va_arg(varg_ptr, int);
				if (n < 0) {
					(void) fprintf(stderr, "panel_set: illegal margin size\n");
					return(LX_ERROR);
				}
				p->xp_rmargin= n;
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXP_BOTTOMMARGIN:
				n= va_arg(varg_ptr, int);
				if (n < 0) {
					(void) fprintf(stderr, "panel_set: illegal margin size\n");
					return(LX_ERROR);
				}
				p->xp_bmargin= n;
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXP_PCURSOR:
				p->xp_pcursor= va_arg(varg_ptr, Cursor);
				XDefineCursor(p->xp_dpy, p->xp_pwin, p->xp_pcursor);
				break;
			case LXP_VSCURSOR:
				p->xp_vscursor= va_arg(varg_ptr, Cursor);
				XDefineCursor(p->xp_dpy, p->xp_vswin, p->xp_vscursor);
				break;
			case LXP_HSCURSOR:
				p->xp_hscursor= va_arg(varg_ptr, Cursor);
				XDefineCursor(p->xp_dpy, p->xp_hswin, p->xp_hscursor);
				break;
			case LXP_VISIBLE:
				n= va_arg(varg_ptr, int);
				if (n) {
					if (!(p->xp_flags & LXP_PANELVISIBLE)) {
						p->xp_flags|= LXP_PANELVISIBLE;
						reconfig= TRUE;
						redisplay= TRUE;
					}
				}
				else {
					if (p->xp_flags & LXP_PANELVISIBLE) {
						p->xp_flags&= ~LXP_PANELVISIBLE;
						undisplay= TRUE;
					}
				}
				break;
			case LXP_CLIENTDATA:
				p->xp_clientdata= va_arg(varg_ptr, char *);
				break;
			case LXP_PROC:
				p->xp_proc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXP_GRAB:
				n= va_arg(varg_ptr, int);
				if (n)
					p->xp_flags|= LXP_GRABPTR;
				else
					p->xp_flags&= ~LXP_GRABPTR;
				break;
			case LXP_ACTIVE:
				n= va_arg(varg_ptr, int);
				if (n) {
					p->xp_flags|= LXP_PANELACTIVE;
					XSelectInput(p->xp_dpy, p->xp_pwin, (ExposureMask | ButtonPressMask | ButtonReleaseMask | ButtonMotionMask | KeyPressMask | PropertyChangeMask));
					XSelectInput(p->xp_dpy, p->xp_vswin, (ExposureMask | ButtonPressMask));
					XSelectInput(p->xp_dpy, p->xp_hswin, (ExposureMask | ButtonPressMask));
				}
				else {
					p->xp_flags&= ~LXP_PANELACTIVE;
					XSelectInput(p->xp_dpy, p->xp_pwin, ExposureMask | PropertyChangeMask);
					XSelectInput(p->xp_dpy, p->xp_vswin, ExposureMask);
					XSelectInput(p->xp_dpy, p->xp_hswin, ExposureMask);
				}
				break;
			case LXP_ALARMINTERVAL:
				n= va_arg(varg_ptr, int);
				if (n < 0) {
					(void) fprintf(stderr, "panel_set: illegal alarm interval\n");
					return(LX_ERROR);
				}
				p->xp_alarmint= n;
				break;
			case LXP_ALARMPROC:
				p->xp_alarmproc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXP_VWIDTH:
			case LXP_VHEIGHT:
			case LXP_VSCROLL:
			case LXP_HSCROLL:
			case LXP_DWIDTH:
			case LXP_DHEIGHT:
				(void) fprintf(stderr, "panel_set: attribute is query-only\n");
				break;
			default:
				(void) fprintf(stderr, "panel_set: ignoring unrecognized attribute\n");
				break;
			}
		}
	}
	va_end(varg_ptr);

	if (undisplay) {
#ifdef PRE_R3
		/* this causes a server crash under X.V11R3/SunOS3.4! */
		XUnmapSubwindows(p->xp_dpy, p->xp_win);
#else
		panel_unmapsubwins(p);
#endif PRE_R3
		return(LX_SUCCESS);
	}

	if (reconfig) {
		if (visible)
#ifdef PRE_R3
			/* this causes a server crash under X.V11R3/SunOS3.4! */
			XUnmapSubwindows(p->xp_dpy, p->xp_win);
#else
			panel_unmapsubwins(p);
#endif PRE_R3
		(void) panel_resize(p);
		(void) panel_config(p);
		p->xp_flags|= LXP_CONFIGDONE;
	}

	if (redisplay) {
		XMapWindow(p->xp_dpy, p->xp_pwin);
		panel_draw(p);
		panel_display(p);
		if (p->xp_flags & LXP_VSCROLLVIS) {
			XMapWindow(p->xp_dpy, p->xp_vswin);
			panelvscroll_draw(p);
		}
		if (p->xp_flags & LXP_HSCROLLVIS) {
			XMapWindow(p->xp_dpy, p->xp_hswin);
			panelhscroll_draw(p);
		}
	}

	return(LX_SUCCESS);
}

int
panel_resize(p)
/*
   Internal function.
   Determines the minimum size necessary
   for the display of the panel.
*/
Panel *p;
{
	Panel_item *pi;

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panel_resize: null panel\n");
		return(LX_ERROR);
	}
	p->xp_mpw= p->xp_mph= 0;

	/* compute minimum size necessary to contain items */
	for (pi= p->xp_items; pi != (Panel_item *) NULL; pi= pi->xpi_next) {
		if ((panelitem_proctab[pi->xpi_type].xpi_sz_proc)(p, pi) != LX_SUCCESS)
			continue;
		if (pi->xpi_x+pi->xpi_w > p->xp_mpw)
			p->xp_mpw= pi->xpi_x+pi->xpi_w;
		if (pi->xpi_y+pi->xpi_h > p->xp_mph)
			p->xp_mph= pi->xpi_y+pi->xpi_h;
	}

	/* add margins or increase to requested size if necessary */
	if (!(p->xp_flags & LXP_REQWIDTH))
		p->xp_mpw+= p->xp_rmargin;
	else if ((p->xp_flags & LXP_REQWIDTH) && (p->xp_mpw < p->xp_rpw))
		p->xp_mpw= p->xp_rpw;
	if (!(p->xp_flags & LXP_REQHEIGHT))
		p->xp_mph+= p->xp_bmargin;
	else if ((p->xp_flags & LXP_REQHEIGHT) && (p->xp_mph < p->xp_rph))
		p->xp_mph= p->xp_rph;

	return(LX_SUCCESS);
}

int
panel_destroy(p)
/*
   User-callable.
   Destroys a panel.
*/
Panel *p;
{
	Panel *xp, *yp;
	Panel_item *xpi, *ypi;
	void panel_unmapsubwins();
	void lxt_clearsel();

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panel_destroy: null panel\n");
		return(LX_ERROR);
	}
	if (p->xp_magic != LX_PANEL) {
		(void) fprintf(stderr, "panel_destroy: object is not a panel\n");
		return(LX_ERROR);
	}

	/* unlink panel from chain */
	if (p == xp_panels)
		xp_panels= xp_panels->xp_next;
	else {
		for (xp= yp= xp_panels; xp != (Panel *) NULL; yp= xp, xp= xp->xp_next) {
			if (xp == p) {
				yp->xp_next= xp->xp_next;
				break;
			}
		}
		if (xp == (Panel *) NULL) {
			(void) fprintf(stderr, "panel_destroy: cannot locate panel\n");
			return(LX_ERROR);
		}
	}

#ifdef PRE_R3
	/* this causes a server crash under X.V11R3/SunOS3.4! */
	XUnmapSubwindows(p->xp_dpy, p->xp_win);
#else
	panel_unmapsubwins(p);
#endif PRE_R3
	XDestroySubwindows(p->xp_dpy, p->xp_win);

	if (p->xp_name != (char *) NULL)
		cfree((char *) p->xp_name);
	if (p->xp_fontnm != (char *) NULL)
		cfree((char *) p->xp_fontnm);

	if (p->xp_ppm != None)
		XFreePixmap(p->xp_dpy, p->xp_ppm);
	if (p->xp_vspm != None)
		XFreePixmap(p->xp_dpy, p->xp_vspm);
	if (p->xp_hspm != None)
		XFreePixmap(p->xp_dpy, p->xp_hspm);
	if (p->xp_stipplepm != None)
		XFreePixmap(p->xp_dpy, p->xp_stipplepm);

	if (p->xp_gc != None)
		XFreeGC(p->xp_dpy, p->xp_gc);
	if (p->xp_igc != None)
		XFreeGC(p->xp_dpy, p->xp_igc);
	if (p->xp_cgc != None)
		XFreeGC(p->xp_dpy, p->xp_cgc);
	if (p->xp_sgc != None)
		XFreeGC(p->xp_dpy, p->xp_sgc);

	if (p->xp_vscroll != (Scrollbar *) NULL)
		cfree((char *) p->xp_vscroll);
	if (p->xp_hscroll != (Scrollbar *) NULL)
		cfree((char *) p->xp_hscroll);

	for (xpi= p->xp_items; xpi != (Panel_item *) NULL; xpi= ypi) {
		ypi= xpi->xpi_next;

		/* make sure that xpi is not involved in any selections */
		if (xpi == (Panel_item *) lxt_selsrc)
			lxt_clearsel(FALSE);
		if (xpi == (Panel_item *) lxt_seldest) {
			lxt_seldest= (Void *) NULL;
			lxt_seldesttype= LX_NULL;
		}

		(void) panelitem_destroy(xpi);
	}

	cfree((char *) p);
	return(LX_SUCCESS);
}

Void *
panel_get(p, attr)
/*
   User-callable routine.
   Returns a pointer to the value of the
   supplied attribute or to the attribute itself.
*/
Panel *p;
int attr;
{
	xpret_void= (Void *) NULL;
	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panel_get: null panel\n");
		return(xpret_void);
	}
	if (p->xp_magic != LX_PANEL) {
		(void) fprintf(stderr, "panel_get: object is not a panel\n");
		return(xpret_void);
	}
	if (attr == LXP_NULL)
		return(xpret_void);

	switch (attr) {

	case LXP_FONT:
		xpret_void= (Void *) p->xp_fontnm;
		break;
	case LXP_FOREGROUND:
		xpret_void= (Void *) &(p->xp_fg);
		break;
	case LXP_BACKGROUND:
		xpret_void= (Void *) &(p->xp_bg);
		break;
	case LXP_WIDTH:
		xpret_void= (Void *) &(p->xp_rpw);
		break;
	case LXP_HEIGHT:
		xpret_void= (Void *) &(p->xp_rph);
		break;
	case LXP_BARWIDTH:
		xpret_void= (Void *) &(p->xp_vscroll->xs_barwidth);
		break;
	case LXP_BUBBLEMARGIN:
		xpret_void= (Void *) &(p->xp_vscroll->xs_bubblemargin);
		break;
	case LXP_BUTTONLEN:
		xpret_void= (Void *) &(p->xp_vscroll->xs_buttonlen);
		break;
	case LXP_RIGHTMARGIN:
		xpret_void= (Void *) &(p->xp_rmargin);
		break;
	case LXP_BOTTOMMARGIN:
		xpret_void= (Void *) &(p->xp_bmargin);
		break;
	case LXP_PCURSOR:
		xpret_void= (Void *) &(p->xp_pcursor);
		break;
	case LXP_VSCURSOR:
		xpret_void= (Void *) &(p->xp_vscursor);
		break;
	case LXP_HSCURSOR:
		xpret_void= (Void *) &(p->xp_hscursor);
		break;
	case LXP_VISIBLE:
		if (p->xp_flags & LXP_PANELVISIBLE)
			xpret_bool= TRUE;
		else
			xpret_bool= FALSE;
		xpret_void= (Void *) &xpret_bool;
		break;
	case LXP_CLIENTDATA:
		xpret_void= (Void *) p->xp_clientdata;
		break;
	case LXP_PROC:
		xpret_void= (Void *) p->xp_proc;
		break;
	case LXP_GRAB:
		if (p->xp_flags & LXP_GRABPTR)
			xpret_bool= TRUE;
		else
			xpret_bool= FALSE;
		xpret_void= (Void *) &xpret_bool;
		break;
	case LXP_VWIDTH:
		(void) panel_resize(p);
		xpret_int= p->xp_mpw+(2*p->xp_ibw);
		xpret_void= (Void *) &xpret_int;
		break;
	case LXP_VHEIGHT:
		(void) panel_resize(p);
		xpret_int= p->xp_mph+(2*p->xp_ibw);
		xpret_void= (Void *) &xpret_int;
		break;
	case LXP_ACTIVE:
		if (p->xp_flags & LXP_PANELACTIVE)
			xpret_bool= TRUE;
		else
			xpret_bool= FALSE;
		xpret_void= (Void *) &xpret_bool;
		break;
	case LXP_ALARMINTERVAL:
		xpret_void= (Void *) &(p->xp_alarmint);
		break;
	case LXP_ALARMPROC:
		xpret_void= (Void *) p->xp_alarmproc;
		break;
	case LXP_VSCROLL:
		if (p->xp_flags & LXP_VSCROLLVIS)
			xpret_bool= TRUE;
		else
			xpret_bool= FALSE;
		xpret_void= (Void *) &xpret_bool;
		break;
	case LXP_HSCROLL:
		if (p->xp_flags & LXP_HSCROLLVIS)
			xpret_bool= TRUE;
		else
			xpret_bool= FALSE;
		xpret_void= (Void *) &xpret_bool;
		break;
	case LXP_DWIDTH:
		xpret_void= (Void *) &(p->xp_apw);
		break;
	case LXP_DHEIGHT:
		xpret_void= (Void *) &(p->xp_aph);
		break;
	default:
		(void) fprintf(stderr, "panel_get: unrecognized attribute\n");
		break;
	}
	return(xpret_void);
}
