/*	tifftomac.c - tiff file to MacPaint converter

	Written by Bill Spitzak
	Besides converting tiff files to MacPaint, this program prints
	all the fields of the file, and provides some useful code for
	examining .tiff files.

*/

#include "port.h"

typedef struct {
	short byteorder;	/* 4949=LSB first, 4D4D=MSB first */
	short version;		/* 42 */
	long ifdoffset;	/* first IFD directory */
	} tiffheader;

/*	an ifd looks like this:
	short count;
	ifdentry entries[count];
	long ifdoffset; (next ifd)
*/

typedef struct {
	short tag;		/* name of the field as a number */
	short type;		/* type of data */
	long length;		/* number of elements */
	long value;		/* data or offset to data if >4 bytes */
	} ifdentry;

/* types: */
#define TIFFBYTE 1
#define TIFFASCII 2
#define TIFFSHORT 3
#define TIFFLONG 4
#define TIFFRATIONAL 5	/* 2 longs, (double)L1/(double)L2 */

#define TIFFSIZEOF(t)	("\1\1\1\2\4\10"[t])

struct {short tag; char *name;} tagtable[] = {
	{254,	"NewSubfileType"},
	{255,	"SubfileType(OLD)"},
	{256,	"ImageWidth"},
	{257,	"ImageLength"},
	{258,	"BitsPerSample"},
	{259,	"Compression"},
	{262,	"PhotometricInterpretation"},
	{263,	"Threshholding(OLD)"},
	{264,	"CellWidth(OLD)"},
	{265,	"CellLength(OLD)"},
	{266,	"FillOrder(OLD)"},
	{269,	"DocumentName"},
	{270,	"ImageDescription"},
	{271,	"Make"},
	{272,	"Model"},
	{273,	"StripOffsets"},
	{274,	"Orientation(OLD)"},
	{277,	"SamplesPerPixel"},
	{278,	"RowsPerStrip"},
	{279,	"StripByteCounts"},
	{280,	"MinSampleValue(OLD)"},
	{281,	"MaxSampleValue(OLD)"},
	{282,	"XResolution"},
	{283,	"YResolution"},
	{284,	"PlanarConfiguration"},
	{285,	"PageName"},
	{286,	"XPosition"},
	{287,	"YPosition"},
	{288,	"FreeOffsets(OLD)"},
	{289,	"FreeByteCounts(OLD)"},
	{290,	"GrayResponseUnit"},
	{291,	"GrayResponseCurve"},
	{292,	"Group3Options"},
	{293,	"Group4Options"},
	{296,	"ResolutionUnit"},
	{297,	"PageNumber"},
	{301,	"ColorResponseCurves"},
	{305,	"Software"},
	{306,	"DateTime"},
	{315,	"Artist"},
	{316,	"HostComputer"},
	{317,	"Predictor"},
	{318,	"WhitePoint"},
	{319,	"PrimaryChromaticities"},
	{320,	"ColorMap"},
	{0,0}};

void *map;			/* points to mapped tiff file */

/*	Get integer constants out of table.  Returns garbage if an offset
	in in the directory! */
int tifflookup(void *table,int tag,int dflt)
{
	int j;
	ifdentry *e;
	long *p;
	for (j = *(short *)table, e = (ifdentry *)(table+2); j--; e++)
		if (e->tag == tag) {
			if (e->length*TIFFSIZEOF(e->type)<=4) p = &e->value;
			else p = map+e->value;
			if (e->type==TIFFSHORT) return(*(short *)p);
			return(*p);
			}
	return(dflt);
	}

/*	returns pointer to object in tiff file */
int tiffoffset(void *table,int tag)
{
	int j;
	ifdentry *e;
	for (j = *(short *)table, e = (ifdentry *)(table+2); j--; e++)
		if (e->tag == tag) {
			if (e->length*TIFFSIZEOF(e->type)>4) return(e->value);
			else return((void *)&e->value-map);
			}
	return(0);
	}

char *tiffname;

/* fixed page size in MacPaint file: */
#define BYTESLINE (576/8)
#define LINESPAGE 720

packwrite(FILE *f,char *b,int l) {
	char *run;
	while (l>0) {
		run = b;
		while (l && *b==*run && b-run<128) {b++; l--;}
		if (b-run > 2) {
			fputc(-(b-run-1),f);
			fputc(*run,f);
			}
		else {
			fputc(b-run-1,f);
			while (run<b) {fputc(*run,f); run++;}
			}
		}
	}

/* library routines needed: */

int min(a,b) register int a, b; {return((a>b)? b : a);}

char *getname(char *cptr)
	{
	char *stop;
	stop=cptr+strlen(cptr);
	while (--stop>=cptr && *stop!='/');
	return(stop+1);
	}

long
filelength(int handle)
	{
	long ltmp, ltmp2;
	ltmp=lseek(handle,0L,1);
	ltmp2=lseek(handle,0L,2);
	lseek(handle,ltmp,0);
	return(ltmp2);
	}

char *setext(char *name,char *ext,int force)
	{
	char *cptr, *c1;
	cptr = c1 = name + strlen(name);
	while (--cptr > name) {
		if (*cptr == '.') {if (!force) return(name); c1 = cptr; break;}
		if (*cptr == '/') break;
		}
	*c1++ = '.';
	strcpy(c1,ext);
	return(name);
	}

/****/

printbitmap(void *table) {
	FILE *f;
	int x,y;
	char *p,*q,fname[80];
	char line[BYTESLINE];
	int BitsPerSample = tifflookup(table,258,1);
	int SamplesPerPixel = tifflookup(table,277,1);
	int ImageWidth = tifflookup(table,256,0);
	int ImageLength = tifflookup(table,257,0);

	strcpy(fname,(char *)getname(tiffname)); setext(fname,"PNTG",TRUE);
	f = fopen(fname,"w");

	/* write empty MacPaint header: */
	for (x=0; x<512; x++) fputc(0,f);

	p = (char *)(map+tifflookup(table,273,0));
	if (ImageLength > LINESPAGE) ImageLength = LINESPAGE;
	for (y=0; y<ImageLength; y++) {
		x = min((ImageWidth+7)/8,BYTESLINE);
		if (BitsPerSample > 1) {
			int n,m,z;
			char *p1 = p;
			for (q=line; q<line+x;) {
				z = 0;
				for (n=7,m=8-BitsPerSample; n>=0; n--) {
					if (n>m) z |= ((~*p)&(1<<m))<<(n-m);
					else z |= ((~*p)&(1<<m))>>(m-n);
					if ((m-=BitsPerSample)<0) {p++; m = 8-BitsPerSample;}
					}
				*q++ = z;
				}
			p = p1+(BitsPerSample*ImageWidth+7)/8;
			}
		else for (q=line; q<line+x;) *q++ = ~(*p++);
		fill(line+x,BYTESLINE-x,0);
		packwrite(f,line,BYTESLINE);
		}
	for (; y<LINESPAGE; y++) {
		fill(line,BYTESLINE,0);
		packwrite(f,line,BYTESLINE);
		}
	fclose(f);
	}

main(int argc,char **argv)
{
	int fd; int i,j,x,y,anum;
	tiffheader *h;
	ifdentry *e;
	void *v;

	if (argc<2) {
		printf("Print the \"directory entries\" of a .tiff file, and\n"
			  "write the bitmap as a MacPaint format file.\n");
		exit(1);
		}
	for (anum=1; anum<argc; anum++) {
	tiffname = argv[anum];
	printf("%s:\n",tiffname);
	fd = open(tiffname,INPUT);
	if (fd<0) {
		printf("Can't open '%s', %s.\n",tiffname,strerror());
		exit(1);
		}
	map_fd(fd,0,&map,TRUE,filelength(fd));
	h = (tiffheader *)map;
	printf("Header: byte order %x, version %d, 1st ifd %x\n\n",
		  h->byteorder,h->version,h->ifdoffset);
	for (i=h->ifdoffset; i;) {
		putchar('\n');
		for (j = *(short *)(map+i),e = (ifdentry *)(map+i+2); j--; e++) {
			for (x=0; tagtable[x].tag && tagtable[x].tag != e->tag; x++);
			if (tagtable[x].tag) printf("%s: ",tagtable[x].name);
			else printf("Tag %d:",e->tag);
			y = e->length;
			if (y<=0) {printf(" Length is %d!\n",y); continue;}
			x = TIFFSIZEOF(e->type)*y;
			if (x<=4) v = &e->value; else v = map+e->value;
			switch(e->type) {
			case TIFFBYTE:
				while (y--) printf(" %02x",*((unsigned char *)v)++);
				break;
			case TIFFASCII:
				printf(" \"%s\"",(char *)v);
				break;
			case TIFFSHORT:
				while (y--) printf(" %d",*((short *)v)++);
				break;
			case TIFFLONG:
				while (y--) printf(" %x",*((long *)v)++);
				break;
			case TIFFRATIONAL:
				while (y--) {
					double a = *((long *)v)++;
					printf(" %g",a/(double)(*((long *)v)++));
					}
				break;
			default: printf("Bad type %d.",e->type); break;
				}
			putchar('\n');
			}
		printbitmap(map+i);
		i = *(long *)e;
		}
	}}
