;;; NeXT Scorefile Package Version 3A

;; Note: This code currently assumes a case-insensitive Lisp.

;; the Zetalisp loop macro (see the Symbolics documentation)
(require 'loop)

;; create the scorefile package
(provide 'scorefile)
(in-package 'scorefile :nicknames '(sc) :use '(lisp user))

(export '(play part update-part score header write-score unique-notetag
          cycle seq env env2 lookup lookup2 ran ranf randomf 
          select pval lastval now notenum timer rhythm rhythms & pitches
          init-random-state accumulate-durs firstnote
          run-program
          *score-file-player* *compile-parameters*))

;;; *** SPECIAL VARIABLES ***

;; a counter for automatically-generated notetags
(defvar *notetag-counter* 0)

;; a counter for automatically-generated part names
(defvar *partname-counter* 0)

;; the name of the last score file created, to be a default for (play)
(defvar *last-score-file* nil)

;; the name of the program used to play score files.  Must take standard input.
(defvar *score-file-player* "playscore")

;;; This variable is set to the score object that is instantiated by the SCORE
;;; macro.  It is used as the default score when PART is called.
(defvar *last-created-score* nil)

;;; If T, compile the parameter definitions into functions, otherwise, just
;;; recursively macroexpand them, then run them interpreted later.
(defvar *compile-parameters* nil)

;;; These may be the final element of a parameter form to indicate output string formatting.
(defconstant format-keywords '(:quiet :string :int :float :str :i :q :db))

;;; These variables are bound to the current part and parameter when a parameter's 
;;; expression is being evaluated.
(proclaim '(special part param))

;;; generate a unique notetag
(defun unique-notetag ()
  (incf *notetag-counter* 1))

;;; run the score file player in a separate process.  ".score" is default extension
(defun play (&optional (file *last-score-file*) (player *score-file-player*)) 
  (setq *last-score-file* 
    (enough-namestring (merge-pathnames file "x.score")))
  (excl:shell (format nil "~a ~a" player *last-score-file*)))

;;; run a program
(defun run-program (arg)
  (excl:shell arg))

;;; Returns true if it's the first note of part
(defmacro firstnote (&optional otherpart)
  `(= 1 (notenum (or ,otherpart part))))

;;; Make it easy to set timeTag from dur (or other parameters)
(defmacro accumulate-durs (&optional (apar 'dur))
  `(quote (if (= (notenum) 1) 0 (+ (lastval) (lastval ,apar)))))

;;; Recursively macroexpands a form. We need this because all macros in a parameter's
;;; expression must be expanded when the parameter is being set up, due to special
;;; macros like cycle and seq.
(defun macroexpand-all (f)
  (cond
   ((atom f) f)
   ((eq (car f) 'quote) f)
   (t
    (let ((ff nil) (f2 (macroexpand f)))
      (loop
        (unless (consp f2) (return))
        (push (macroexpand-all (pop f2)) ff))
      (nreverse ff)))))


;;; return the actual format string indicated by the keyword.
(defun format-key (keyword)
  (case (or keyword :float)
    ((:string :str) "~s")
    (:float "~0,3f")
    ((:int :i) "~d")
    (:db "-~,1fdB")
    (:q :quiet)
    (t keyword)))


;;; *** STRUCTURE DEFINITIONS ***

;;; The Parameter structure.  There is one parameter instance created for each
;;; specified parameter in a part.  Note that accessor functions will have a "param-"
;;; prefix, for brevity.

(defstruct (parameter (:conc-name param-))
  ;; the string name of the parameter which will appear in the score file
  name
  ;; the part instance to which the parameter belongs
  part
  ;; the string used by format to print the value of the parameter in the score
  format-string
  ;; whether or not to omit the parameter in a phrase if the value has not changed
  (optimize t)
  ;; the function which computes the value of the parameter
  expr
  ;; the current value of the parameter
  current-value
  ;; the previous value of the parameter
  previous-value
  ;; association list of notetags (if any) and the previous value for each notetag
  notetag-values)

;;; The Part structure, which defines a score file part 

(defstruct part
  ;; the string name of the part as it will appear in the score file
  name
  ;; a list of other partinfo parameters and values
  partinfo
  ;; the score object to which the part belongs
  score
  ;; the normal part that an update part is updating
  parent-part
  ;; the notetag - computed automatically if not set
  tag
  ;; part status, one of :initialized, :note, :rest, :end, or :done
  status
  ;; the previous note's status
  previous-status
  ;; association list of notetags used by the part and the last status at each notetag
  notetag-status
  ;; the part's begin time relative to the score's begin time
  (begin-time 0)
  ;; the part's cutoff time relative to the score's begin time
  cutoff-time
  ;; the current time in beats since the beginning of the part
  note-time
  ;; the parameter which determines the begin time of a note
  time-parameter
  ;; the parameter which determines the duration of a note
  dur-parameter
  ;; the parameter which determines the notetag, if any
  notetag-parameter
  ;; type of note, one of (:noteDur :noteOn :update :noteoff)
  note-type
  ;; current note number since beginning of part
  (note-num 0)
  ;; an alist of parameter name strings and instances
  parameters
  ;; a list of all non-parameter objects (like seq) being used by the part's parameters
  instances
  ;; update-parts declared within this part - used only during initialization
  subparts)

;;; The score structure, which is a collection of parts, sorted by note-time

(defstruct score
  ;; string file name for the score
  (name "test.score")
  ;; the score info parameters, if any
  scoreinfo
  ;; a list of all part objects in the score
  parts
  ;; a string to be included in the header (e.g., envelopes and wavetables)
  (header "")
  ;; whether to use relative or absolute times in the score file
  (use-relative-times t)
  ;; the begin time of the first note of the score
  (begin-time 0)
  ;; when to cutoff the score (optional)
  (cutoff-time most-positive-fixnum)
  ;; time of the last note output to the score file
  (last-note-time 0)
  ;; time of the current note to be output
  (current-note-time 0)
  ;; note number of this note
  (note-num 0))

;;; *** SCORE, PART, AND PARAMETER CREATION ***

;;; Creates a new parameter.  Either fully macroexpands
;;; the given form or (if non-constant) creates a function and compiles it. 
;;; Adds the new parameter instance to the part's parameter list.
(defun parameter (part name form &rest keywords)
  (let ((par (make-parameter :part part 
                             :optimize (not (find :always keywords))))
        (pname (string name)))
     (declare (special part))
     (setf (param-format-string par) 
       (format-key (first (intersection format-keywords keywords))))
     (setf (param-expr par) 
       (if (consp form)
           (if *compile-parameters* 
               (compile nil (list 'lambda '() form))
             (macroexpand-all form))
         form))
     (setf (param-name par) pname)
     ;; add new param to end of list to preserve evaluation order
     (setf (part-parameters part) 
       (nconc (part-parameters part) (list (cons pname par))))
     ;; make note of certain special parameters
     (cond ((string-equal pname "timetag")
            (setf (part-time-parameter part) par)
            (setf (param-format-string par) :quiet)
            )
           ((string-equal pname "dur")
            (setf (part-dur-parameter part) par)
            (setf (param-format-string par) :quiet)
            )
           ((string-equal pname "notetag")
            (setf (part-notetag-parameter part) par)
            (setf (param-format-string par) :quiet))
           (t nil))
     par))

;;; Creates a special part for updating another part.  This is called within Part
;;; for nested update declarations.  It may also be called on its own.
(defmacro update-part (parent-part &optional (times 0) &rest parameters)
  `(let* ((parent ,parent-part)
          (part (make-part
                 :name (part-name parent)
                 :parent-part parent
                 :score (part-score parent)
                 :note-type :update
                 :begin-time (+ (if (consp ,times) (first ,times) (or ,times 0.0))
                                (part-begin-time parent))
                 :cutoff-time (if (consp ,times) 
                                  (+ (second ,times) 
                                     (part-begin-time parent)))
                 )))
     (declare (special 'part))
     ;; process each parameter or update form
     ,(append '(progn)
              (loop for pn in parameters
                for name = (first pn)
                do (when (and (symbolp name) (not (boundp name)))
                     (set (intern name 'user) (string-downcase name))
                     (proclaim (list 'special name)))
                collect (if (eq (first pn) 'update)
                            `(update-part part ,@(cdr pn))
                          `(parameter part ,@pn))))
     ;; initialize all the parameters to their first value
     (push part (part-subparts parent))
     ;; if parent has already been initialized, initialize this update part
     ;; Updates defined within a part definition are initialized by the main part
     (when (part-status parent) 
       (advance-parameters part)
       (when *last-created-score* (add-part *last-created-score* part)))
     part))

(defun break-if-case-sensitive ()
  (if (not (eq 'foo 'FOO))
      (block foo
        (terpri)
        (princ "Warning: The NeXT Scorefile Package assumes a case-insensitive-upper Lisp.")
        (terpri)
        (princ "The Lisp you are running is case-insensitive.")
        (terpri)
        (princ "See your system administrator for help.")
        (terpri)
        (terpri)
        (break)
        )
    )
  )

;;; parses a part definition and creates a new part instance.  uses the PARAMETER
;;; function to parse all the parameters.  Adds the instance to the score's list
;;; of parts
(defmacro part (name partinfo &optional (times 0) (notetype :noteOn) &rest parameters)
  `(let* ((part (make-part
                 :name (cond ((and ',name (symbolp ',name)) (string-downcase ',name))
                             ((or (consp ',name) (stringp ,name))
                              (string-downcase ,name))
                             (t (format nil "part~d" (incf *partname-counter* 1))))
                 :partinfo ',partinfo
                 :score *last-created-score*
                 :note-type ,notetype
                 :begin-time (if (consp ,times) (first ,times) ,times)
                 :cutoff-time (if (consp ,times) (second ,times))))
          (globalvar (intern (string-upcase (string (part-name part))) 'user)))
     (declare (special 'part))
     (set globalvar part)
     (proclaim (list 'special globalvar))
     ;; process all parameter and update forms.
     ,(append '(progn)
              (loop for pn in parameters
                for name = (first pn)
                do (when (and (symbolp name) (not (boundp name)))
                     (set (intern name 'user) (string-downcase name))
                     (proclaim (list 'special name)))
	       collect (if (eq (first pn) 'update)
                            `(update-part part ,@(cdr pn))
                          `(parameter part ,@pn))))
     ;; initialize all the parameters to their first value
     (init-part-and-subparts part)
     part))

;;; get the named parameter instance
(defun getpar (par &optional otherpart)
  (declare (special part))
  (cdr (assoc par (part-parameters (or otherpart part))
              :test #'string-equal)))

;; This routine insures that parameters are initialized to their first values
;; in a top-down fashion.  That is, sub-parts (updates) will be initialized
;; after their parent parts so that they may reference the parent's parameter
;; values from the beginning.
(defun INIT-PART-AND-SUBPARTS (part)
  (setf (part-status part) :initialized)
  ;; initialize this part's parameters.
  (advance-parameters part)
  ;; create a unique notetag for :noteOn part types if one has not been specified.
  (if (and (eq (part-note-type part) :noteon) (not (part-notetag-parameter part)))
      (setf (part-tag part) (unique-notetag)))
  (when *last-created-score* (add-part *last-created-score* part))
  ;; now initialize any update parts
  (loop for pt in (part-subparts part) do 
    (init-part-and-subparts pt)))

;;; creates a new score object.  Will also add any given parts which were
;;; created without a score, or for a different score.
(defmacro score (&optional name scoreinfo (times 0) 
                 (use-relative-times t) header &rest parts)
  `(let ((sc (make-score
                :begin-time (if (consp ,times) (first ,times) (or ,times 0.0))
                :cutoff-time (if (consp ,times) (second ,times) 
                               most-positive-single-float)
                :scoreinfo ',scoreinfo
                :header ,header
                :use-relative-times ,use-relative-times))
         str)
     (break-if-case-sensitive)
     (setf (score-last-note-time sc) (score-begin-time sc))
     (setf (score-current-note-time sc) (score-begin-time sc))
     (setq *last-created-score* sc)
     (loop for one-part in ,parts
           do (add-part sc one-part))
     (when ',name 
       (set (intern 
             (setq str (string-upcase 
                        (cond ((symbolp ',name) ',name)
                              ((or (consp ',name) (stringp ,name)) 
                               ,name))))
             'user) sc)
       (setf (score-name sc) (merge-pathnames (string-downcase str) "x.score")))
     sc)) 


;;; *** PARAMETER METHODS ***

;;; returns t if the parameter expression is to be considered a constant value.
(defun constant-expr-p (expr)
  (or (numberp expr) (symbolp expr) (stringp expr)))

;;; sets the current value to the result of funcalling the parameter's function,
;;; if there is one.
(defun next-par-value (param)
  (let ((expr (param-expr param)))
    (setf (param-previous-value param) (param-current-value param))
    (setf (param-current-value param)
      (if (constant-expr-p expr) expr
        (let ((part (param-part param)))
          (catch 'parameter-expr-tag 
            (if (functionp expr) (funcall expr) (eval expr))))))))


;;; *** PART METHODS ***

;;; resets attributes of a part so that it can be re-run.
(defun reset-part (part)
  (setf (part-note-time part) (part-begin-time part))
  (setf (part-status part) :initialized)
  (setf (part-previous-status part) nil)
  (setf (part-notetag-status part) nil)
  (setf (part-note-num part) 0)
  (loop for par in (part-parameters part) do
    (setf (param-previous-value (cdr par)) nil)
    (setf (param-notetag-values (cdr par)) nil))
  (loop for i in (part-instances part) do (reset-parameter-form i)))

;;; calls next-par-value on all the parameters.  Returns nil if all are ok,
;;; :end if somebody returned an :end keyword, or :rest if a parameter returned
;;; :r or the duration parameter was negative.  Computes next note time.
(defun advance-parameters (part)
  (let* ((sc (part-score part))
	 (durpar (part-dur-parameter part))
	 (timepar (part-time-parameter part))
	 (tagpar (part-notetag-parameter part))
	 (tag (part-tag part))
	 (status (part-status part))
	 (save-notetag-values (and tagpar tag (not (eq (part-note-type part) :notedur))))
	 tmp)
    (declare (special part))
    (setf (part-previous-status part) status)
    (if (and tagpar tag)
	;; remember the previous status relative to previous notetag
	(if (setq tmp (assoc tag (part-notetag-status part)))
	    (rplacd tmp status)
	  (push (cons tag status) (part-notetag-status part))))
    (if (and (> (incf (part-note-num part) 1) 1)
             (not timepar))
        (setq status :end)
      (setq status :note))
    (loop for pair in (part-parameters part)
          for par = (cdr pair)
          for val = (next-par-value par)
       when save-notetag-values
         do (if (setq tmp (assoc tag (param-notetag-values par))) ; tag is for last note
		(rplacd tmp (param-previous-value par))
	      (pushnew (cons tag (param-previous-value par)) (param-notetag-values par)))
       when (eq val :end) 
         do (setq status :end)
       when (and timepar (eq par timepar) (not (eq val :end)) (< val 0))
         do (setf (param-current-value timepar) (- val))
         and do (setq val :r)
       when (and durpar (eq par durpar) (not (eq val :end)) (< val 0))
         do (setf (param-current-value durpar) (- val))
         and do (setq val :r)
       when (and (eq val :r) (not (eq status :end)))
         do (setq status :rest))
    (setf (part-note-time part)
      (if (and timepar (numberp (param-current-value timepar)))
          (+ (part-begin-time part)
             (if timepar (param-current-value timepar) 0))
        (if (not (eq status :end))
            (part-begin-time part)
          (if (part-dur-parameter part)
              (+ (part-note-time part)
                 (param-current-value (part-dur-parameter part)))
            (or (part-cutoff-time part)
                (+ (or (part-note-time part) (part-begin-time part)) 0.25))))))
    (when (and timepar (numberp (param-current-value timepar))
               (or (and (part-cutoff-time part)
                        (>= (part-note-time part) (part-cutoff-time part)))
                   (>= (part-note-time part) (score-cutoff-time sc))))
      (setq status :end))
    (when tagpar 
      (setf (part-tag part) (param-current-value tagpar)))
    (setf (part-status part) status)))

(defmacro PREVIOUS-STATUS (part tag)
  `(cdr (assoc ,tag (part-notetag-status ,part))))

(defmacro PREVIOUS-VALUE (param tag)
  `(cdr (assoc ,tag (param-notetag-values ,param))))

;;; prints a note to the given stream.
(defun print-note (part &optional (stream t))
  (let* ((sc (part-score part))
         (len 24)
         (*print-case* :downcase)
         (curtime (score-current-note-time sc))
         (lasttime (score-last-note-time sc))
         (type (part-note-type part))
         (timepar (part-time-parameter part))
         (durpar (part-dur-parameter part))
	 (tagpar (part-notetag-parameter part))
         (tag (part-tag part)))
    ;; omit noteupdates in which nothing has changed.
    (if (and (eq type :update)
             (loop for pair in (part-parameters part)
                   for par = (cdr pair)
                   always (or (eq par timepar)
                              (equal (param-current-value par) 
				     (if tagpar
					 (previous-value par tag)
				       (param-previous-value par))))))
        (return-from print-note nil))
    (format t "~8,3f" curtime)
    (incf (score-note-num sc) 1)
    (if (score-use-relative-times sc)
        (format stream "~&t +~,5f;" (- curtime lasttime))
      (format stream "~&t ~,5f;" curtime))
    (format stream "~&~a " (part-name part))
    (case type
          (:noteDur 
           (let ((durval (or (if durpar (param-current-value durpar)) 0.25)))
             (if tag
                 (format stream "(~0,3f,~d)" durval tag)
               (format stream "(~0,3f)" durval))))
          (:noteOn
           (unless tag
             (error "NIL noteTag for noteOn in part ~a" (part-name part)))
           (format stream "(noteOn,~d)" tag))
          (:noteoff
           (unless tag
             (error "NIL noteTag for noteOff in part ~a" (part-name part)))
           (format stream "(noteOff,~d);" tag))
          (:update
           (if tag
               (format stream "(noteUpdate,~d)" tag)
             (princ "(noteUpdate)" stream))))
    (unless (eq type :noteoff)
      (loop for pair in (part-parameters part)
            for par = (cdr pair)
            when (and (not (eq (param-format-string par) :quiet))
                      (or (not (constant-expr-p (param-expr par))) 
                          (not (param-optimize par)) 
                          (eq type :update)))
            do (let* ((value (param-current-value par))
                      str)
                 (when (and value
                            (not (and (not (eq type :notedur))
				      (eq (if tagpar
					      (previous-status part tag)
					    (part-previous-status part)) :note)
                                      (param-optimize par)
                                      (equal value
					     (if tagpar
						 (previous-value par tag)
					       (param-previous-value par))))))
                   (setq str (format nil (concatenate 'string " ~a:" 
                                                      (param-format-string par))
                                     (param-name par) value))
                   (setq len (+ len (length str)))
                   (when (>= len 80) 
                     (format stream "~%~8T")
                     (setq len (+ 8 (length str))))
                   (write-string str stream))))
      (write-char #\; stream))
    (if (= (mod (score-note-num sc) 10) 0) (terpri))
    nil))

;;; prints the non-tagged noteupdate which contains a part's static parameters.
(defun print-noteupdate (part &optional (stream t))
  (let ((len 24)
        (*print-case* :downcase)
        (pars (part-parameters part)))
    (format stream "~&~a (noteUpdate)" (part-name part))
    (loop for pair in pars
          for par = (cdr pair)
          when (and (not (eq (param-format-string par) :quiet)) 
                    (and (constant-expr-p (param-expr par))))
          do (let* ((value1 (param-current-value par))
                    (value (if (and (symbolp value1) (boundp value1))
                               (eval value1) value1))
                    (str (format nil (concatenate 'string " ~a:" 
                                                  (param-format-string par))
                                 (param-name par) value)))
               (setq len (+ len (length str)))
               (when (> len 80) 
                     (format stream "~%~8T")
                     (setq len (+ 8 (length str))))
               (write-string str stream)))
    (loop for par in (part-partinfo part)
          for name = (eval (if (consp par) (first par) par))
          unless (or (not (consp par))
                     (string-equal name "synthpatchcount")
                     (string-equal name "synthpatch")
                     (getpar name part))
          do (let ((str (format nil 
                                (concatenate 'string " ~a:" 
                                             (or (format-key (third par)) "~0,3f"))
                                name (eval (second par)))))
               (setq len (+ len (length str)))
               (when (> len 80) 
                     (format stream "~%~8T")
                     (setq len (+ 8 (length str))))
               (write-string str stream)))
    (write-char #\; stream)
    nil))
  
;;; *** SCORE METHODS ***

;;; reset attributes of a score to allow for recomputation
(defun reset-score (sc)
  (loop for pt in (score-parts sc) do (reset-part pt))
  (setf (score-current-note-time sc) (score-begin-time sc))
  (setf (score-note-num sc) 0)
  (setf (score-header sc) nil))

;;; add a string to the header string of a score file
(defun header (str &optional (sc *last-created-score*))
  (setf (score-header sc) (concatenate 'string (score-header sc) (string str))))

;;; returns t if first part should come before second part.  Insures that
;;; sub-parts will be run after their parent-parts if note times are the same.
(defun part-less-than (part1 part2)
  (let ((t1 (part-note-time part1))
	(t2 (part-note-time part2)))
    (or (< t1 t2)
	(and (= t1 t2) (not (part-parent-part part1)) (part-parent-part part2)))))

;;; assign a part object to a score
(defun add-part (sc part)
  (unless (member part (score-parts sc))
          (setf (score-parts sc) 
            (merge 'list (score-parts sc) (list part) #'part-less-than))))

;;; remove a part object from a score
(defun remove-part (sc part)
  (setf (score-parts sc) (delete part (score-parts sc))))

;;; compute and print the next note of the next part due for a new note.
(defun generate-note (sc stream)
  (let* ((parts (score-parts sc))
         (pt (first parts))
         (type (part-note-type pt)))
    (setf (score-last-note-time sc) (score-current-note-time sc))
    (setf (score-current-note-time sc) 
      (max (+ (score-begin-time sc) (part-note-time pt))
           (+ (score-current-note-time sc) .00002)))
    (when pt
      (case (part-status pt)
            (:end
	     ;; cutoff reached or :end seen
             (progn
               (when (and (eq type :noteOn) (> (part-note-num pt) 0))
                     (setf (part-note-type pt) :noteoff)
		     (if (part-notetag-parameter pt)
			 (dolist (tag-status (part-notetag-status pt))
			   (when (eq (cdr tag-status) :note)
			     (setf (part-tag pt) (car tag-status))
			     (print-note pt stream)))
		       (print-note pt stream)))
               (setf (part-note-time pt) most-positive-single-float)
               (setf (part-status pt) :done)))
            (:rest
	     ;; negative duration or :r seen
             (if (and (= (part-note-num pt) 1) (not (eq type :update)))
                 (print-noteupdate pt stream))
	     ;; only make noteOff if necessary - check last status
             (when (and (eq type :noteOn)
			(if (part-notetag-parameter pt)
			    (eq (previous-status pt (part-tag pt)) :note)
			  (eq (part-previous-status pt) :note)))
	       (setf (part-note-type pt) :noteoff)
	       (print-note pt stream)
	       (setf (part-note-type pt) :noteOn)))
            (t
	     ;; a regular note
             (if (and (= (part-note-num pt) 1) (not (eq type :update)))
                 (print-noteupdate pt stream))
             (print-note pt stream)))
      ;; advance the parameters for this part, then update its position in the queue.
      (unless (eq (part-status pt) :done)
        (advance-parameters pt))
      (setf (score-parts sc) (sort parts #'part-less-than)))
    (not (eq (part-status (first (score-parts sc))) :done))))
      
;;; compute and write out the score file. 
(defun write-score (sc &optional filename-or-stream)
  (if (stringp filename-or-stream) (setf (score-name sc) filename-or-stream))
  (let* ((existing-stream (streamp filename-or-stream))
         (stream (cond ((or existing-stream (eq filename-or-stream t)) 
                        filename-or-stream)
                       ((score-name sc)
                        (open (score-name sc) :direction :output 
                              :if-does-not-exist :create
                              :if-exists :new-version))
                       (t t)))
         (info (score-scoreinfo sc)))
    (setq *notetag-counter* 0)
    (setq *partname-counter* 0)
    (if (> (score-note-num sc) 0) (score-reset sc))
    ;; first print the score info
    (when info
      (format stream "info ")
      (loop for i in info do
        (format stream (concatenate 'string " ~a:" 
                                    (or (format-key (third i)) "~0,3f"))
                (eval (first i)) (eval (second i))))
      (format stream ";~%"))
    (unwind-protect
        (let ((*print-case* :downcase))
	  ;; print the part declarations
          (loop for pt in (score-parts sc) 
                unless (eq (part-note-type pt) :update)
                do (format stream "part ~a;~%" (part-name pt)))
	  ;; print the part info
          (loop for pt in (score-parts sc) 
                for info = (part-partinfo pt)
                unless (eq (part-note-type pt) :update)
                do (format stream "~a" (part-name pt))
                and do
                  (when (stringp (first info))
                    (format stream " synthPatch:~s" (first info))
                    (setq info (cdr info)))
                and do
                  (loop for par in info do
                    (if (consp par)
                        (format stream 
                                (concatenate 'string " ~a:" 
                                             (or (format-key (third par)) "~0,3f"))
                                (eval (first par)) (eval (second par)))))
                and do (format stream ";~%"))
          (terpri stream)
	  ;; print the score header
          (if (score-header sc) (write-string (score-header sc) stream) )
          (format stream "~%~%BEGIN;~%")
          (if (score-use-relative-times sc)
              (format stream "t ~,5f;~%" (score-begin-time sc)))
	  ;; here's the main loop
          (loop while (generate-note sc stream))
          (format stream "~%END;~%"))
      (if (or existing-stream (eq stream t))
          (force-output stream)
        (close stream)))
    (setq *last-score-file* (score-name sc))
    (terpri)(force-output)
    ))


;;;  *** PARAMETER VALUE GENERATING STRUCTURES AND METHODS ***

;;; Parameter value generating structures and functions.  Each structure should have
;;; a structure definition which will contain its state, a <structurename>-value
;;; function which returns a value for the note, and a macro which the user will call.
;;; The macro will typically create an instance, push it onto the parts
;;; instances list, and return a form which calls the value function on the
;;; instance created.   The generic structure "parameter-form" has some
;;; standard slots which are set to nil by the generic reset function.
;;; The special variable "part" is always bound to the current part.

(defstruct parameter-form
  current-values
  original-values)

(defun reset-parameter-form (param-form)
  (setf (parameter-form-current-values param-form) nil)
  (setf (parameter-form-original-values param-form) nil))

;;; CYCLE: Loop throught the list of values.  Keep looping unless ":end" is seen.
;;; The values may be constants or expressions.

(defstruct (cycle (:include parameter-form)))

(defun next-cycle-value (c &optional values)
  (let ((vals (or (cycle-current-values c) (cycle-original-values c)))
        val)
    (unless vals 
            (setq vals values)
            (setf (cycle-current-values c) values)
            (setf (cycle-original-values c) values))
    (setq val (eval (first vals)))
    (setf (cycle-current-values c) (cdr vals))
    (if (eq val :end) (throw 'parameter-expr-tag :end))
    (if (eq val :r) (throw 'parameter-expr-tag :r))
    val))

(defmacro cycle (values &optional instance)
  (let ((var (gentemp 'music)))
    (declare (special part))
    (set var (or (eval instance) (make-cycle)))
    (pushnew (eval var) (part-instances part))
    `(locally (declare (special ,var))
              (values (next-cycle-value ,var ,values) ,var))))


;;; SEQ: Step through the list of values specified. Stick on last value.
;;; The values may be constants or expressions

(defstruct (seq (:include parameter-form)))

(defun next-seq-value (sq &optional values)
  (let* ((vals (or (seq-current-values sq) values))
         (val (eval (first vals))))
    (if (eq val :end)
        (throw 'parameter-expr-tag :end)
      (if (second vals)
          (setf (seq-current-values sq) (cdr vals))))
    (if (eq val :r) (throw 'parameter-expr-tag :r))
    val))

(defmacro seq (values &optional instance)
  (let ((var (gentemp 'music)))
    (declare (special part))
    (set var (or (eval instance) (make-seq)))
    (pushnew (eval var) (part-instances part))
    `(locally (declare (special ,var))
              (values (next-seq-value ,var ,values) ,var))))


;;; NOW: returns the current time in beats since the beginning of the part
(defun now (&optional otherpart)
  (declare (special part))
  (let ((timepar (part-time-parameter (or otherpart part))))
    (if timepar (param-current-value timepar) 0)))

;;; NOTENUM: returns the current note number of the part
(defmacro notenum (&optional otherpart)
  `(part-note-num (or ,otherpart part)))

;;; LOOKUP: lookup and interpolate a value given an arbitrary index
;;; Return first y value if index is less than first envelope x, or last value if
;;; index is greater than last envelope x.  Envelope is in form ((x1 y1)(x2 y2)...).
;;; The x values must be in ascending order.

(defun eval-car (expr)
  (eval (car expr)))

(defun lookup (index values)
  (let ((i1 (if (numberp index) 
                (position index values :key #'eval-car :test #'>= :from-end t)
              (unless (or (eq index :r) (eq index :end))
                (error "Non-numeric index ~a to lookup or env" index)))))
    (cond ((not i1)
           (eval (cadar values)))
          ((= i1 (- (length values) 1))
           (eval (cadar (last values))))
          (t
           (let* ((xy1 (nth i1 values))
                  (xy2 (nth (+ i1 1) values))
                  (x1 (eval (first xy1)))
                  (y1 (eval (second xy1))))
             (+ y1 (* (- (eval (second xy2)) y1) 
                      (/ (- index x1) (- (eval (first xy2)) x1)))))))))

;;; ENV: return interpolated Y-value of envelope indexed by current part note time.
(defmacro env (values)
  `(lookup (now) ,values))

;;; LOOKUP2: lookup and return an envelope value given an arbitrary index.
;;; Return first y value whose corresponding x value is less than or equal to the
;;; given index, or the first y value if the index is less than the first x value.
;;; No interpolation is done.

(defun lookup2 (index values)
  (eval (second 
         (nth (or (if (numberp index) 
                      (position index values :key #'eval-car 
                                :test #'>= :from-end t)) 0)
                 values))))

;;; ENV2: return non-interpolated Y-value of envelope indexed by current 
;;; part note time, using lookup2.
(defmacro env2 (values)
  `(lookup2 (now) ,values))
 
;;; RANDOMF: return a sequence of correlated noise values.  Correlation factor
;;; is 0 for white noise, 1 for the last value, and in between for something
;;; close to the last value.  Values in the .5 - .8 range are useful.  Like
;;; the lisp random function, returns a value of the same type as the range
;;; argument.

(defstruct randomf lastvalue)

(defun next-randomf-value (rf correlation range &optional start)
  (let* ((lastvalue (or (randomf-lastvalue rf) start (* range .5)))
         (tmp (* (- 1.0 correlation) range))
         (low (max (- lastvalue tmp) 0))
         (high (min (+ lastvalue tmp) range)))
    (setf (randomf-lastvalue rf) (+ low (random (- high low))))
    (if (typep range 'integer) 
        (floor (randomf-lastvalue rf)) 
      (randomf-lastvalue rf))))
  
(defmacro randomf (correlation range start &optional instance)
  (let ((var (gentemp 'music)))
    (set var (or (eval instance) (make-randomf)))
    `(locally (declare (special ,var))
              (values (next-randomf-value ,var ,correlation ,range ,start) ,var))))


;;; *** OTHER UTILITY FUNCTIONS AND MACROS ***

;;; RAN: Returns randomly selected members of a list.
(defun ran (values)
  (nth (random (length values)) values))


;;; RANF: Returns randomly selected members of a list using RANDOMF (see above)
(defmacro ranf (correlation values init)
  `(eval (nth (floor (randomf ,correlation (length ,values) ,init)) ,values)))


;;; SELECT: Returns the (0-based) index into a list of values.
(defmacro select (index values)
  `(nth (floor ,index) ,values))


;;; PVAL: returns the value of the named parameter.  The optional second argument
;;; may be used to get the current value of a paramter from another part.  The
;;; first argument must be a string or an unquoted form which evaluates to a string
;;; naming some previously-defined parameter.
(defun pval (param &optional otherpart)
  (let* ((par (getpar param otherpart))
         (pt (or otherpart part))
         (val nil))
    (loop while (and (not par) (setq pt (part-parent-part pt))) do
      (setq par (getpar param pt)))
    (if par (setq val (param-current-value par)))
    (if (eq val :r) (throw 'parameter-expr-tag :r))
    val))

;;; LASTVAL: like pval, but gets the previous note's value
(defun lastval (&optional otherparam otherpart)
  (let* ((param (if otherparam (getpar otherparam otherpart) param))
         (val (if param (param-previous-value param))))
    (if (eq val :r) (throw 'parameter-expr-tag :r))
    val))

(defmacro timer (&rest args)
  (let ((end (- (length args) 1)))
    (append '(cond)
            (loop for i from 0 to end collect
                (list (if (< i end) 
                          `(< (now) ,(first (nth (+ i 1) args))) t)
                      (second (nth i args)))))))

;;; Some functions for using lists of rhythmic values instead of durations.

;;; converts the argument, a single rhythmic value, into a duration
(defun rhythm (single-rhythm)
  (if (numberp single-rhythm) (/ 4.0 single-rhythm) single-rhythm))

;;; converts the arguments, which are rhythms, into a list of durations.
(defun rhythms (&rest rhythm-list)
  (mapcar #'rhythm rhythm-list))

(defun tie-fn (x y)
  (+ x (/ 4.0 y)))

;;; produces a single rhythmic value equivalent to the argument rhythmic
;;; values tied together
(defun & (&rest rhythm-list)
  (/ 4.0 (reduce #'tie-fn rhythm-list :initial-value 0)))

;;; Replaces a list of unquoted pitch names in which octave numbers are assumed
;;; to be "sticky" with a list of quoted pitch names, all with octave numbers,
;;; suitable for passing to seq or cycle.
(defmacro pitches (&rest pitch-symbols)
  (let ((old-octave 4)
        (new-octave 4))
    (list 'quote
          (loop 
           for p in pitch-symbols
           for str = (symbol-name p)
           do (setq new-octave 
                (parse-integer str :start (- (length str) 1) :junk-allowed t))
           collect 
            (cond ((eq p :end) :end)
                  ((eq p :r) :r)
                  (t
                   (list 'quote 
                         (if new-octave 
                             (progn (setq old-octave new-octave) p)
                           (find-symbol 
                            (format nil "~a~d" (string-upcase str) 
                                    old-octave) 'sc)))))))))

;;; a kludge to allow a C-like random initialization, since CL doesn't provide it.
;;; *random-state* is a common-lisp predefined special variable, used by RANDOM.
(defun init-random-state (seed)
  (setq *random-state* 
    (read-from-string (format nil "#s(RANDOM-STATE :SEED ~d)" seed))))

;;; First all the Music Kit parameter names in all their case-sensitive glory

(defconstant parnames
  '(
    "_noPar"
    "keyPressure"
    "afterTouch"
    "controlChange"
    "pitchBend"
    "programChange"
    "timeCodeQ"
    "songPosition"
    "songSelect"
    "tuneRequest"
    "sysExclusive"
    "chanMode"
    "sysRealTime"

    "basicChan"
    "controlVal"
    "monoChans"

    "velocity"
    "relVelocity"
    "keyNum"

    "velocitySensitivity"
    "afterTouchSensitivity"
    "modWheelSensitivity"
    "breathSensitivity"
    "footSensitivity"
    "portamentoSensitivity"
    "balanceSensitivity"
    "panSensitivity"
    "expressionSensitivity"

    "freq"
    "amp"
    "bearing"
    "bright"
    "portamento"
    
    "waveform"
    "waveLen"
    "phase"

    "cRatio"
    "c2Ratio"
    "c2Amp"
    "c2Waveform"
    "c2Phase"
    "c3Ratio"
    "c3Amp"
    "c3Waveform"
    "c3Phase"
    "m1Ratio"
    "m1Ind"
    "m1Waveform"
    "m1Phase"
    "m2Ratio"
    "m2Ind"
    "m2Waveform"
    "m2Phase"
    "m3Ratio"
    "m3Ind"
    "m3Waveform"
    "m3Phase"
    "m4Ratio"
    "m4Ind"
    "m4Waveform"
    "m4Phase"
    "feedback"
    
    "pickNoise"
    "decay"
    "sustain"
    "lowestFreq"

    "svibFreq"
    "svibAmp"
    "rvibFreq"
    "rvibAmp"
    "indSvibFreq"
    "indSvibAmp"
    "indRvibFreq"
    "indRvibAmp"
    
    "noiseAmp"
    "noiseFreq"

    "freqEnv"
    "freq0"
    "freqAtt"
    "freqRel"
    "ampEnv"
    "amp0"
    "ampAtt"
    "ampRel"
    "bearingEnv"
    "bearing0"
    "brightEnv"
    "bright0"
    "brightAtt"
    "brightRel"
    "waveformEnv"
    "waveform0"
    "waveformAtt"
    "waveformRel"

    "c2AmpEnv"
    "c2Amp0"
    "c2AmpAtt"
    "c2AmpRel"
    "c3AmpEnv"
    "c3Amp0"
    "c3AmpAtt"
    "c3AmpRel"
    "m1IndEnv"
    "m1Ind0"
    "m1IndAtt"
    "m1IndRel"
    "m2IndEnv"
    "m2Ind0"
    "m2IndAtt"
    "m2IndRel"
    "m3IndEnv"
    "m3Ind0"
    "m3IndAtt"
    "m3IndRel"
    "m4IndEnv"
    "m4Ind0"
    "m4IndAtt"
    "m4IndRel"

    "svibFreqEnv"
    "svibFreq0"
    "rvibFreqEnv"
    "rvibFreq0"
    "indSvibFreqEnv"
    "indSvibFreq0"
    "indRvibFreqEnv"
    "indRvibFreq0"

    "svibAmpEnv"
    "svibAmp0"
    "rvibAmpEnv"
    "rvibAmp0"
    "indSvibAmpEnv"
    "indSvibAmp0"
    "indRvibAmpEnv"
    "indRvibAmp0"
    
    "noiseAmpEnv"
    "noiseAmp0"
    "noiseAmpAtt"
    "noiseAmpRel"
    "noiseFreqEnv"
    "noiseFreq0"
    
    "synthPatch"
    "synthPatchCount"
    "midiChan"
    "track"

    "title"
    "samplingRate"
    "headroom"
    "tempo"

    "_illegalPar"
    "_dur"

    "freq1"
    "amp1"
    "bright1"
    "bearing1"
    "waveform1"
    "c1Ratio"
    "c1Amp"
    "c1AmpAtt"
    "c1AmpRel"
    "c1Waveform"
    "phase1"
    "c1Amp1"
    "c2Amp1"
    "c3Amp1"
    "m1Ind1"
    "m2Ind1"
    "m3Ind1"
    "m4Ind1"
    "svibFreq1"
    "rvibFreq1"
    "indSvibFreq1"
    "indRvibFreq1"
    "svibAmp1"
    "rvibAmp1"
    "indSvibAmp1"
    "indRvibAmp1"
    "noiseAmp1"
    "noiseFreq1"

    "dur"
    "timetag"
    "notetag"
    "update"
    ))

;;; creates a constant symbol with the given string as a print-name,
;;; the same string as its value.
(defun make-constant (str)
  (let ((var (intern (string-upcase str) 'scorefile)))
    (eval `(defconstant ,var ,str))
    (export var)))

;;; make constants out of all the MK parameter strings.
(defun initialize-parnames ()
  (loop for param in parnames 
        do (make-constant param)))

(initialize-parnames)

;;; midi controller numbers
(defconstant midi-modwheel           1)
(defconstant midi-breath             2)
(defconstant midi-foot               4)
(defconstant midi-portamentotime     5)
(defconstant midi-dataentry          6)
(defconstant midi-mainvolume         7)
(defconstant midi-balance            8)
(defconstant midi-pan                10)
(defconstant midi-expression         11)
;;; lsb for above
(defconstant midi-modwheellsb        (+ 1 31)  )
(defconstant midi-breathlsb          (+ 2 31))
(defconstant midi-footlsb            (+ 4 31))
(defconstant midi-portamentotimelsb  (+ 5 31))
(defconstant midi-dataentrylsb       (+ 6 31))
(defconstant midi-mainvolumelsb      (+ 7 31))
(defconstant midi-balancelsb         (+ 8 31))
(defconstant midi-panlsb             (+ 10 31))
(defconstant midi-expressionlsb      (+ 11 31))
;;; midi switch numbers
(defconstant midi-damper             64)
(defconstant midi-portamento         65)
(defconstant midi-sostenuto          66)
(defconstant midi-softpedal          67)
(defconstant midi-hold2              69)
(defconstant midi-externaleffectsdepth 91)
(defconstant midi-tremelodepth       92)
(defconstant midi-chorusdepth        93)
(defconstant midi-detunedepth        94)
(defconstant midi-phaserdepth        95)
(defconstant midi-dataincrement      96)
(defconstant midi-datadecrement      97)

(export '(midi-modwheel midi-breath midi-foot midi-portamentotime midi-dataentry
          midi-mainvolume midi-balance midi-pan midi-expression midi-modwheellsb
          midi-breathlsb midi-footlsb midi-portamentotimelsb midi-dataentrylsb
          midi-mainvolumelsb midi-balancelsb midi-panlsb midi-expressionlsb
          midi-damper midi-portamento midi-sostenuto midi-softpedal midi-hold2
          midi-externaleffectsdepth midi-tremelodepth midi-chorusdepth 
          midi-detunedepth midi-phaserdepth midi-dataincrement midi-datadecrement
          init-freqs))

;;; create variables out of the pitch names from c0 to b9, and initialize them
;;; to equal temperament.
(defun init-freqs ()
  (let ((middle-c (/ 440 (expt 2 (/ 9 12)))))
    (loop for i from 0 to 9
     do (loop 
         for note in '(c cs d ds e f fs g gs a as b)
         for j from 0 to 11
         for equiv in '(bs df eff ef ff es gf aff af bff bf cf)
         for sym = (intern (format nil "~a~d" note i) 'scorefile)
         for equiv-sim = (intern (format nil "~a~d" equiv i) 'scorefile)
         do 
          (set sym (* middle-c (expt 2.0 (/ j 12.0)) 
                      (expt 2.0 (- i 4.0))))
          (set equiv-sim (if (< j 11) (eval sym) (/ (eval sym) 2.0)))
          (proclaim (list 'special sym equiv-sim))
          (export (list sym equiv-sim))))))

(init-freqs)

;;; rhythmic values; whole, triplet whole, half, triplet half, etc.
;;; suitable for passing to the RHYTHMS function.

(defconstant  w 1.0)
(defconstant tw 1.5)
(defconstant  h 2.0)
(defconstant th 3.0)
(defconstant  q 4.0)
(defconstant tq 6.0)
(defconstant  e 8.0)
(defconstant te 12.0)
(defconstant  s 16.0)
(defconstant ts 24.0)
(defconstant  k 32.0)
(defconstant tk 48.0)

(export '(w tw h th q tq e te s ts k tk))

(use-package 'scorefile)
