#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/tcfs_fs.h>
#include <linux/tcfsiod.h>
#include <linux/malloc.h>
#include <linux/pagemap.h>

#include <asm/segment.h>
#include <asm/system.h>

#ifdef DEBUG_BIO
#define dprintk(args...)	printk(## args)
#else
#define dprintk(args...)	/* nothing */
#endif

#include <linux/tcfs/cipherinterface.h>
/* These are the function needed to handle keys hash table */
#include <linux/tcfs/hash.h>
/*extern struct hash_entry ** hash_table;
extern void init_hash(void);
extern struct hash_entry * hash_add(int uid,char *deskey, void *other_data);
extern struct hash_entry * hash_lookup(int uid);
extern void hash_rem(int uid);*/


static inline int
do_read_tcfs_sync(struct inode * inode, struct page * page)
{
	struct tcfs_fattr fattr;
	int		result, refresh = 0;
	int		count = PAGE_SIZE;
	int		rsize = TCFS_SERVER(inode)->rsize;
	char		*buf = (char *) page_address(page);
	unsigned long	pos = page->offset;
	
	dprintk("TCFS: do_read_tcfs_sync(%p)\n", page);

	set_bit(PG_locked, &page->flags);
	clear_bit(PG_error, &page->flags);

	do {
		if (count < rsize)
			rsize = count;
		result = tcfs_proc_read(TCFS_SERVER(inode), TCFS_FH(inode), 
			pos, rsize, buf, &fattr);
		dprintk("tcfs_proc_read(%s, (%x,%lx), %ld, %d, %p) = %d\n",
				TCFS_SERVER(inode)->hostname,
				inode->i_dev, inode->i_ino,
				pos, rsize, buf, result);
		/*
		 * Even if we had a partial success we can't mark the page
		 * cache valid.
		 */
		if (result < 0)
			goto io_error;
		refresh = 1;
		count -= result;
		pos += result;
		buf += result;
		if (result < rsize)
			break;
	} while (count);

	memset(buf, 0, count);
	set_bit(PG_uptodate, &page->flags);
	result = 0;

io_error:
	if (refresh /*&& jiffies-TCFS_READTIME(inode)>=TCFS_ATTRTIMEO(inode)*/) {
		tcfs_refresh_inode(inode, &fattr);
		if (S_ISREG(inode->i_mode) && 
			(TCFS_IS_SHARED(inode) || 
			TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag)) && 
			inode->i_size>0 && inode->u.tcfs_i.tcfs_fl.bf.spure>0) {
				inode->i_size-=8-inode->u.tcfs_i.tcfs_fl.bf.spure;
	}
	}
	clear_bit(PG_locked, &page->flags);
	wake_up(&page->wait);
	return result;
}

/*
 * This is the function to (re-) transmit a TCFS readahead request
 */
static int
tcfsiod_read_setup(struct tcfsiod_req *req)
{
	struct inode	*inode = req->rq_inode;
	struct page	*page = req->rq_page;

	return tcfs_proc_read_request(&req->rq_rpcreq,
			TCFS_SERVER(inode), TCFS_FH(inode),
			page->offset, PAGE_SIZE, 
			(__u32 *) page_address(page));
}

/*
 * This is the callback from tcfsiod telling us whether a reply was
 * received or some error occurred (timeout or socket shutdown).
 */
static int
tcfsiod_read_result(int result, struct tcfsiod_req *req)
{
	struct tcfs_server *server = TCFS_SERVER(req->rq_inode);
	struct page	*page = req->rq_page;
	static int	succ = 0, fail = 0;
	int		i;

	dprintk("BIO: received callback for page %p, result %d\n",
			page, result);

	if (result >= 0) {
		struct tcfs_fattr	fattr;

		result = tcfs_proc_read_reply(&req->rq_rpcreq, &fattr);
		if (result >= 0) {
			tcfs_refresh_inode(req->rq_inode, &fattr);
			if (result < PAGE_SIZE)
				memset((u8 *) page_address(page)+result,
						0, PAGE_SIZE-result);
		}
	} else
	if (result == -ETIMEDOUT && !(server->flags & TCFS_MOUNT_SOFT)) {
		/* XXX: Theoretically, we'd have to increment the initial
		 * timeo here; but I'm not going to bother with this now
		 * because this old tcfsiod stuff will soon die anyway.
		 */
		result = -EAGAIN;
	}

	if (result == -EAGAIN && req->rq_retries--) {
		dprintk("BIO: retransmitting request.\n");
		memset(&req->rq_rpcreq, 0, sizeof(struct tcrpc_ioreq));
		while (tcrpc_reserve(server->rsock, &req->rq_rpcreq, 1) < 0)
			schedule();
		current->fsuid = req->rq_fsuid;
		current->fsgid = req->rq_fsgid;
		for (i = 0; i < NGROUPS; i++)
			current->groups[i] = req->rq_groups[i];
		tcfsiod_read_setup(req);
		return 0;
	}
	if (result >= 0) {
		set_bit(PG_uptodate, &page->flags);
		succ++;
	} else {
		dprintk("BIO: %d successful reads, %d failures\n", succ, fail);
		set_bit(PG_error, &page->flags);
		fail++;
	}
	clear_bit(PG_locked, &page->flags);
	wake_up(&page->wait);
	free_page(page_address(page));
	return 1;
}

static inline int
do_read_tcfs_async(struct inode *inode, struct page *page)
{
	struct tcfsiod_req *req;
	int		result, i;

	dprintk("TCFS: do_read_tcfs_async(%p)\n", page);

	set_bit(PG_locked, &page->flags);
	clear_bit(PG_error, &page->flags);

	if (!(req = tcfsiod_reserve(TCFS_SERVER(inode))))
		return -EAGAIN;

	req->rq_retries = 5;
	req->rq_callback = tcfsiod_read_result;
	req->rq_inode = inode;
	req->rq_page = page;

	req->rq_fsuid = current->fsuid;
	req->rq_fsgid = current->fsgid;
	for (i = 0; i < NGROUPS; i++)
		req->rq_groups[i] = current->groups[i];

	if ((result = tcfsiod_read_setup(req)) >= 0) {
		page->count++;
		tcfsiod_enqueue(req);
	} else {
		dprintk("TCFS: deferring async READ request.\n");
		tcfsiod_release(req);
		clear_bit(PG_locked, &page->flags);
		wake_up(&page->wait);
	}

	return result < 0? result : 0;
}

int
tcfs_readpage(struct inode *inode, struct page *page)
{
	unsigned long	address=0,i;
	int		error = -1;
	struct hash_entry *htmp=NULL;
	struct gid_hash_entry *ghtmp=NULL;
	void *ks=NULL;
	char doit=0;

	if (TCFS_IS_SHARED(inode)) {
		ghtmp=gid_hash_lookup(inode->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			free_page(address);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL || current->uid != inode->i_uid) {
				free_page(address);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}
	dprintk("TCFS: tcfs_readpage %08lx\n", page_address(page));
	address = page_address(page);
	page->count++;
	if (!PageError(page) && TCFS_SERVER(inode)->rsize >= PAGE_SIZE)
		error = do_read_tcfs_async(inode, page);
	if (error < 0)		/* couldn't enqueue */
		error = do_read_tcfs_sync(inode, page);
	if (doit) {
		for (i=0;i<PAGE_SIZE;i+=1024) {
/*			if (inode->i_size-page->offset-i < 1024) {
				tcfs_decrypt((unsigned char *)(address+i),inode->i_size-page->offset-i,ks);
				break;
			} else {*/
				tcfs_decrypt((unsigned char *)(address+i),1024,ks);
/*			}*/
		}
	}
	free_page(address);
	return error;
}
