/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc. and it's affiliated companies
	All rights reserved.

	$Id: PGPsdaMacDecrypt.cp,v 1.7 1999/03/30 05:11:18 heller Exp $
____________________________________________________________________________*/

#include <PP_KeyCodes.h>	/* Needed only for char_Backspace */

#include "MacDialogs.h"
#include "MacEnvirons.h"
#include "MacErrors.h"
#include "MacEvents.h"
#include "MacFiles.h"
#include "MacMenus.h"
#include "MacQuickDraw.h"
#include "MacStrings.h"
#include "pgpMem.h"

#include "CipherContext.h"
#include "CipherProcGlue.h"
#include "lzss.h"
#include "PGPsdaMacCommon.h"
#include "PGPsdaMacDecrypt.h"
#include "PGPsdaMacFileFormat.h"
#include "SHA.h"

#define	kMaxSpacesPerCharacter		2
#define kMaxPassphraseLength		255

#define	kSDAPassphraseDialogResID			1003
#define	kSDAAppearanceProgressDialogResID	1004
#define	kSDAStringListResID					1004
#define	kSDAErrorAlertResID					2103

pgpFixBeforeShip( "Non appearance progress dialog" )


#define	kInputBufferSize			(128L * 1024L)
#define	kOutputBufferSize			(128L * 1024L)


enum
{
	kOKButtonID			= 1,
	kCancelButtonID,
	kPassphraseEditTextID,
	kHideTypingCheckboxID,
	kPromptStaticTextID
};

enum
{
	kNameExtensionStrIndex		= 1,
	kDecryptToPromptStrIndex,
	kIntegrityCheckFailureStrIndex,
	kCorruptedArchiveStrIndex,
	kErrorOccurredStrIndex
};

enum
{
	kStopButtonID			= 1,
	kMessageStaticTextID,
	kProgressBarDialogItemID
};


const uchar	kOptionSpaceCharacter = 0xCA;	// <<<--- This is an option-Space!

typedef struct PGPsdaDecryptionContext
{
	const FSSpec 	*sourceFileSpec;
	const FSSpec 	*outFolderSpec;
	long			outDirID;
	short			archiveFileRef;
	char			passphrase[kMaxPassphraseLength];
	PGPUInt32		remainingDecompressBytes;
	PGPUserValue	userValue;
	CipherProcRef	cipherProcRef;
	CipherContext	cipherContext;
	CASTKey			encryptionKey;
	SDABuffer		inputBuffer;
	SDABuffer		outputBuffer;
	SDABuffer		decompressedBuffer;
	SDABuffer		decryptedBuffer;
	OSStatus		decompressionErr;
	PGPUInt32		decryptedBlockIV;
	SHA				*hash;
	
	PGPsdaArchiveHeader			archiveHeader;
	PGPsdaDecryptProgressProc	progressProc;
	PGPsdaDecryptProgressInfo	progressInfo;
	
} PGPsdaDecryptionContext;

static Boolean 	sPassphraseHidden;
static char		sPassphrase[kMaxPassphraseLength];
static char		sSpacePassphrase[kMaxPassphraseLength * kMaxSpacesPerCharacter];
static UInt32	sPassphraseLength;
static UInt32	sSpacePassphraseLength;
static UInt8	sSpaceCountList[kMaxPassphraseLength];


	static pascal Boolean
PassphraseDialogFilterProc(
	DialogPtr 		theDialog,
	EventRecord 	*theEvent,
	DialogItemIndex *itemHit)
{
	Boolean	handled = FALSE;
	GrafPtr	savePort;
	
	GetPort( &savePort );
	SetPort( theDialog );
	
	switch( theEvent->what )
	{
		case updateEvt:
		{
			RGBColor	saveBackColor;
			Rect		itemRect;
			
			BeginUpdate( theDialog );
			
			EraseRgn( theDialog->visRgn );
			
			GetBackColor( &saveBackColor );
			BackColor( whiteColor );
			
			GetDialogItemRect( theDialog, kPassphraseEditTextID, &itemRect );
			InsetRect( &itemRect, -3, -3 );
			EraseRect( &itemRect );
			
			RGBBackColor( &saveBackColor );
			
			UpdateDialog( theDialog, theDialog->visRgn );
			
			EndUpdate( theDialog );
			break;
		}
		
		case keyDown:
		case autoKey:
		{
			handled = StdFilterProc( theDialog, theEvent, itemHit );
			if( ! handled )
			{
				UInt8		charCode 	= theEvent->message & charCodeMask;
				TEHandle	textH		= ((DialogPeek) theDialog)->textH;
				RGBColor	saveBackColor;

				GetBackColor( &saveBackColor );
				BackColor( whiteColor );
			
				handled = TRUE;
				
				if( charCode >= ' ' )
				{
					UInt32	numSpaces;
					
					numSpaces 				= sSpaceCountList[sPassphraseLength];
					sSpacePassphraseLength	+= numSpaces;
					
					sPassphrase[sPassphraseLength++] = charCode;
					
					if( sPassphraseHidden )
					{
						UInt32	spaceIndex;
						
						// If the passphrase is hidden, add the Spaces
						for( spaceIndex = 0; spaceIndex < numSpaces; spaceIndex++ )
						{
							TEKey( kOptionSpaceCharacter, textH );
						}
					}
					else
					{
						TEKey( charCode, textH );
					}
				}
				else if( charCode == char_Backspace )
				{
					if( (**textH).selEnd > 0 )
					{
						UInt32	backspaceCount;
						UInt32	index;
						UInt32	numSpaces;
						
						pgpAssert( sPassphraseLength > 0 );

						--sPassphraseLength;
						
						numSpaces = sSpaceCountList[sPassphraseLength];
						sSpacePassphraseLength	-= numSpaces;
						
						if( sPassphraseHidden )
						{
							backspaceCount = numSpaces;
						}
						else
						{
							backspaceCount = 1;
						}
						
						for( index = 0; index < backspaceCount; index++ )
						{
							::TEKey( char_Backspace, textH );
						}
					}
				}
				else
				{
					/* Filter low ASCII characters */
					
					SysBeep( 1 );
				}

				RGBBackColor( &saveBackColor );
			}
			
			break;
		}
		
		case mouseDown:
		{
			short	theItem;
			Point	localPt;
			
			localPt = theEvent->where;
			GlobalToLocal( &localPt );
			
			theItem = FindDialogItem( theDialog, localPt );
			if( theItem == kPassphraseEditTextID - 1 )
			{
				handled = TRUE;
			}
		
			break;
		}
	}
	
	if( ! handled )
		handled = StdFilterProc( theDialog, theEvent, itemHit );
	
	SetPort( savePort );
	
	return( handled );
}

	static OSStatus
PGPsdaDecryptPassphraseDialog(char passphrase[kMaxPassphraseLength])
{
	DialogPtr	theDialog;
	GrafPtr		savePort;
	OSStatus	err = noErr;
	
	passphrase[0] = 0;
	
	GetPort( &savePort );
	
	/* Lock down the passphrase because it is sensitive */

	HoldBuffer( sPassphrase, sizeof( sPassphrase ) );

	/*
	** Pre-fill an array of space characters for simple showing and
	** hiding of the passphrase.
	*/
	pgpFillMemory( sSpacePassphrase, sizeof( sSpacePassphrase ),
			kOptionSpaceCharacter );

	/*
	** Generate a "random" amount of space characters to represent the
	** entered character. This is stored so that backspacing appears to
	** work "correctly". The number of spaces is derived using the crappy
	** Macintosh Random() routine.
	*/
		
	for( short index = 0; index < kMaxPassphraseLength; index++ )
	{
		sSpaceCountList[index] =
			( (ulong) Random() % kMaxSpacesPerCharacter ) + 1;
	}

	sPassphraseHidden		= TRUE;
	sPassphraseLength		= 0;
	sSpacePassphraseLength	= 0;

	theDialog = GetNewDialog( kSDAPassphraseDialogResID, NULL, (WindowPtr) -1L );
	if( IsntNull( theDialog ) )
	{
		ModalFilterUPP	filterUPP 	= NewModalFilterProc( PassphraseDialogFilterProc );
		short			itemHit = 0;
		MenuHandle		appleMenu;
		Boolean			enableAppleMenuItem = FALSE;
		
		/*
		** Disabling the first item in the Apple menu will force the
		** Dialog Manager to disable the Edit menu in a modal dialog.
		*/
		
		appleMenu = GetIndMenuInMenuBar( 0 );
		if( IsntNull( appleMenu ) && CountMenuItems( appleMenu ) >= 1 )
		{
			enableAppleMenuItem = IsMenuItemEnabled( appleMenu, 1 );
			if( enableAppleMenuItem )
			{
				DisableItem( appleMenu, 1 );
			}
		}
		
		SetPort( theDialog );
		
		SetDialogDefaultItem( theDialog, kOKButtonID );
		SetDialogCancelItem( theDialog, kCancelButtonID );
		
		SetDialogControlValue( theDialog, kHideTypingCheckboxID, 1 );
		
		ShowWindow( theDialog );
		SelectWindow( theDialog );
		
		do
		{
			ModalDialog( filterUPP, &itemHit );
			
			switch( itemHit )
			{
				case kHideTypingCheckboxID:
				{
					TEHandle	textH = ((DialogPeek) theDialog)->textH;

					sPassphraseHidden = ToggleDialogCheckbox( theDialog,
							kHideTypingCheckboxID );
					if( sPassphraseHidden )
					{
						TESetText( sSpacePassphrase, sSpacePassphraseLength,
								textH );
					}
					else
					{
						TESetText( sPassphrase, sPassphraseLength, textH );
					}
					
					InvalDialogItem( theDialog, kPassphraseEditTextID, 0 );
					break;
				}
				
				case kOKButtonID:
				{
					pgpCopyMemory( sPassphrase, passphrase, sPassphraseLength );
					passphrase[sPassphraseLength] = 0;
					break;
				}
				
				case kCancelButtonID:
					err = userCanceledErr;
					break;
			}
			
		} while( itemHit != kOKButtonID && itemHit != kCancelButtonID );
		
		DisposeRoutineDescriptor( filterUPP );
		DisposeDialog( theDialog );
		
		if( enableAppleMenuItem )
		{
			EnableItem( appleMenu, 1 );
		}
	}
	else
	{
		err = memFullErr;
	}
	
	SetPort( savePort );
	
	UnholdBuffer( sPassphrase, sizeof( sPassphrase ), TRUE );
	
	return( err );
}

	static OSStatus
VerifySDAArchiveHeader(
	const PGPsdaArchiveHeader 	*header,
	PGPUInt32					archiveFileSize)
{
	if( header->magic != kPGPsdaArchiveMagic )
	{
		pgpDebugMsg( "\pInvalid archive header magic" );
		return( kSDAError_InvalidArchive );
	}
	
	if( header->fileSize != archiveFileSize )
	{
		pgpDebugMsg( "\pIncorrect file length" );
		return( kSDAError_InvalidArchive );
	}
	
	return( noErr );
}

	static void
EndSDADecryption(PGPsdaDecryptionContext *context)
{
	if( context->cipherProcRef != NULL )
	{
		CipherProc_Dispose( context->cipherProcRef );
		context->cipherProcRef = NULL;
	}
	
	if( context->archiveFileRef > 0 )
	{
		FSClose( context->archiveFileRef );
		context->archiveFileRef = 0;
	}

	FreeSDABuffer( &context->inputBuffer );
	FreeSDABuffer( &context->outputBuffer );
	FreeSDABuffer( &context->decompressedBuffer );
	FreeSDABuffer( &context->decryptedBuffer );
	
	delete( context->hash );
}

	static OSStatus
BeginSDADecryption(PGPsdaDecryptionContext *context)
{
	OSStatus	err = noErr;
	
	context->decryptedBlockIV 	= 1;
	context->hash				= new( SHA );
	
	/* Read the archive header */
	err = FSpOpenDF( context->sourceFileSpec, fsRdPerm, &context->archiveFileRef );
	if( IsntErr( err ) )
	{
		long	count = sizeof( context->archiveHeader );
		
		err = FSRead( context->archiveFileRef, &count, &context->archiveHeader );
		if( IsntErr( err ) )
		{
			long	fileSize;
			
			GetEOF( context->archiveFileRef, &fileSize );
			
			context->progressInfo.bytesProcessed 		= count;
			context->progressInfo.totalBytesToProcess 	= fileSize;
		
			err = VerifySDAArchiveHeader( &context->archiveHeader, fileSize );
		}
	}
	
	if( IsntErr( err ) && context->archiveHeader.isEncryptedArchive )
	{
		/* Initialize cipher proc */
		
		pgpCopyMemory( context->archiveHeader.passphraseSalt,
				context->cipherContext.salt.saltBytes,
				sizeof( context->cipherContext.salt ) );
				
		HashSaltPassphrase( context->passphrase, &context->cipherContext.salt,
				&context->archiveHeader.hashReps, &context->encryptionKey );

		err = CipherProc_Load( &context->cipherProcRef );
		if( IsntErr( err ) )
		{
			err = CipherProc_CallInitContext( context->cipherProcRef,
						&context->cipherContext,
						(uchar *) &context->encryptionKey );
			if( IsntErr( err ) )
			{
				/* Verify the passphrase */
				
				if( ! VerifyCheckBytes( context->cipherProcRef,
							&context->cipherContext,
							&context->encryptionKey,
							&context->archiveHeader.checkBytes ) )
				{
					err = kSDAError_IncorrectPassphrase;
				}
			}
		}
	}
	
	if( IsntErr( err ) )
	{
		err = AllocateSDABuffer( kInputBufferSize, &context->inputBuffer );
		if( IsntErr( err ) )
		{
			err = AllocateSDABuffer( context->archiveHeader.compressedBlockSize,
						&context->decompressedBuffer );
			if( IsntErr( err ) )
			{
				err = AllocateSDABuffer( context->archiveHeader.compressedBlockSize * 2,
							&context->decryptedBuffer );
				if( IsntErr( err ) )
				{
					err = AllocateSDABuffer( kOutputBufferSize, &context->outputBuffer );
				}
			}
		}
	}
	
	if( IsErr( err ) )
	{
		EndSDADecryption( context );
	}
	
	return( err );
}
	
	static OSStatus
FillInputBuffer(PGPsdaDecryptionContext *context)
{
	OSStatus	err;
	long		count;
	PGPUInt32	existingBytes;
	
	pgpAssert( context->inputBuffer.pos <= context->inputBuffer.used );
	
	existingBytes = context->inputBuffer.used - context->inputBuffer.pos;
	if( existingBytes != 0 )
	{
		/* Move existing bytes to the beginning of the buffer */
		pgpCopyMemory( context->inputBuffer.data + context->inputBuffer.pos,
				context->inputBuffer.data, existingBytes );
	}
	
	context->inputBuffer.used	= existingBytes;
	context->inputBuffer.pos 	= 0;
	count						= context->inputBuffer.size - existingBytes;
	
	err = FSRead( context->archiveFileRef, &count, context->inputBuffer.data +
					 context->inputBuffer.used );
	if( err == eofErr )
	{
		/* Read ahead failed. Let buffer logic handle error*/
		pgpAssert( "Read ahead failed" );
		err = noErr;
	}
	
	if( count != 0 )
	{
		context->hash->Update( context->inputBuffer.data + context->inputBuffer.used,
			count );

		context->inputBuffer.used 				+= count;
		context->progressInfo.bytesProcessed	+= count;	
	}
	
	if( IsntErr( err ) && IsntNull( context->progressProc ) )
	{
		pgpAssert( context->progressInfo.bytesProcessed <= 
				context->progressInfo.totalBytesToProcess );
		
		if( context->progressInfo.bytesProcessed >
					context->progressInfo.totalBytesToProcess )
		{
			context->progressInfo.bytesProcessed = 
					context->progressInfo.totalBytesToProcess;
		}
		
		err = context->progressProc( &context->progressInfo, context->userValue );
	}
	
	return( err );
}

	static OSStatus
FillDecryptedBuffer(PGPsdaDecryptionContext *context)
{
	OSStatus	err = noErr;
	PGPUInt32	remainingBlocks;
	PGPUInt32	availEncryptedBlocks;
	PGPUInt32	existingBytes;
	
	pgpAssert( context->decryptedBuffer.pos <= context->decryptedBuffer.used );
	
	existingBytes = context->decryptedBuffer.used - context->decryptedBuffer.pos;
	if( existingBytes != 0 )
	{
		/* Move existing bytes to the beginning of the buffer */
		pgpCopyMemory( context->decryptedBuffer.data + context->decryptedBuffer.pos,
				context->decryptedBuffer.data, existingBytes );
	}
	
	context->decryptedBuffer.used 	= existingBytes;
	context->decryptedBuffer.pos	= existingBytes;
		
	remainingBlocks 		= ( context->decryptedBuffer.size -
									existingBytes ) / 512L;
	availEncryptedBlocks 	= ( context->inputBuffer.used -
									context->inputBuffer.pos ) / 512L;
	
	while( remainingBlocks != 0 && IsntErr( err ) )
	{
		/* Input buffer is always manipulated using entire blocks */
		
		pgpAssert( ( context->inputBuffer.pos % 512L ) == 0 );
		pgpAssert( ( context->inputBuffer.used % 512L ) == 0 );
		
		if( availEncryptedBlocks != 0 )
		{
			PGPUInt32	blocksToDecrypt;
			PGPUInt32	bytesToDecrypt;
			
			blocksToDecrypt = remainingBlocks;
			if( blocksToDecrypt > availEncryptedBlocks )
				blocksToDecrypt = availEncryptedBlocks;
			
			bytesToDecrypt = blocksToDecrypt * 512L;
			
			if( context->archiveHeader.isEncryptedArchive )
			{
				err = CipherProc_CallDecrypt( context->cipherProcRef,
							&context->cipherContext, context->decryptedBlockIV,
							blocksToDecrypt, context->inputBuffer.data +
							context->inputBuffer.pos, context->decryptedBuffer.data +
							context->decryptedBuffer.pos );
			}
			else
			{
				pgpCopyMemory( context->inputBuffer.data +
						context->inputBuffer.pos, context->decryptedBuffer.data +
						context->decryptedBuffer.pos, bytesToDecrypt );
			}
					
			context->decryptedBuffer.used 	+= bytesToDecrypt;
			context->decryptedBuffer.pos 	+= bytesToDecrypt;
			context->inputBuffer.pos		+= bytesToDecrypt;
			availEncryptedBlocks			-= blocksToDecrypt;
			remainingBlocks					-= blocksToDecrypt;
			context->decryptedBlockIV		+= blocksToDecrypt;
		}
		
		if( availEncryptedBlocks == 0 && remainingBlocks != 0)
		{
			err = FillInputBuffer( context );
			
			availEncryptedBlocks = ( context->inputBuffer.used -
										context->inputBuffer.pos ) / 512L;
			if( availEncryptedBlocks == 0 )
				remainingBlocks = 0;
		}
	}

	context->decryptedBuffer.pos = 0;

	return( err );
}

	static OSStatus
ReadFromDecryptor(
	PGPsdaDecryptionContext *context,
	const void				*data,
	PGPSize					dataSize)
{
	UInt32		bytesRemaining 	= dataSize;
	PGPByte		*curData		= (PGPByte *) data;
	OSStatus	err 			= noErr;
	
	while( bytesRemaining != 0 && IsntErr( err ) )
	{
		PGPUInt32	bytesAvail;
		
		/* Copy bytes to buffer, which may or may not fill the buffer */
		
		bytesAvail = context->decryptedBuffer.used - context->decryptedBuffer.pos;
		if( bytesAvail != 0 )
		{
			PGPUInt32	bytesToCopy = bytesRemaining;
			
			if( bytesToCopy > bytesAvail )
				bytesToCopy = bytesAvail;
				
			pgpCopyMemory( context->decryptedBuffer.data + 
					context->decryptedBuffer.pos, curData, bytesToCopy );
					
			context->decryptedBuffer.pos	+= bytesToCopy;
			curData							+= bytesToCopy;
			bytesRemaining 					-= bytesToCopy;
		}
		
		if( bytesRemaining != 0  )
		{
			pgpAssert( context->decryptedBuffer.pos == context->decryptedBuffer.used );
			
			err = FillDecryptedBuffer( context );
			if( IsntErr( err ) && context->decryptedBuffer.used == 0 )
				err = eofErr;
		}
	}
		
	return( err );
}

	static OSStatus
DecompressBlock(PGPsdaDecryptionContext *context)
{
	OSStatus				err = noErr;
	PGPsdaCompBlockHeader	header;
	
	pgpAssert( context->decompressedBuffer.pos == context->decompressedBuffer.used );

	err = ReadFromDecryptor( context, &header, sizeof( header ) );
	if( IsntErr( err ) )
	{
		/* The size of the buffer to decompress is embedded in the data stream */
		
		pgpAssert( header.magic == 'dave' );
		
		if( header.notCompressed )
		{
			err = ReadFromDecryptor( context, context->decompressedBuffer.data, header.blockSize );
			
			context->decompressedBuffer.used 	= header.blockSize;
			context->decompressedBuffer.pos		= 0;
		}
		else
		{
			context->remainingDecompressBytes 	= header.blockSize; 
			context->decompressedBuffer.pos 	= 0;
			
			lzss_decompress( context );
			
			context->decompressedBuffer.used 	= context->decompressedBuffer.pos;
			context->decompressedBuffer.pos		= 0;
		
			err = context->decompressionErr;
		}
	}

	return( err );
}

	static OSStatus
ReadFromDecompressor(
	PGPsdaDecryptionContext *context,
	const void				*data,
	PGPSize					dataSize)
{
	UInt32		bytesRemaining 	= dataSize;
	PGPByte		*curData		= (PGPByte *) data;
	OSStatus	err 			= noErr;
	
	while( bytesRemaining != 0 && IsntErr( err ) )
	{
		PGPUInt32	bytesAvail;
		
		/* Copy bytes to buffer, which may or may not fill the buffer */
		
		bytesAvail = context->decompressedBuffer.used - context->decompressedBuffer.pos;
		if( bytesAvail != 0 )
		{
			PGPUInt32	bytesToCopy = bytesRemaining;
			
			if( bytesToCopy > bytesAvail )
				bytesToCopy = bytesAvail;
				
			pgpCopyMemory( context->decompressedBuffer.data +
					context->decompressedBuffer.pos, curData, bytesToCopy );
					
			context->decompressedBuffer.pos	+= bytesToCopy;
			curData							+= bytesToCopy;
			bytesRemaining 					-= bytesToCopy;
		}
		
		if( bytesRemaining != 0 )
		{
			pgpAssert( context->decompressedBuffer.pos == context->decompressedBuffer.used );
	
			err = DecompressBlock( context );
			if( IsntErr( err ) && context->decompressedBuffer.used == 0 )
				err = eofErr;
		}
	}
		
	return( err );
}

	int
decompress_putc_buffer(
	int 	invoer,
	void 	*pUserValue)
{
	PGPsdaDecryptionContext	*context = (PGPsdaDecryptionContext *) pUserValue;
	
	if( IsntErr( context->decompressionErr ) )
	{
		pgpAssert( context->decompressedBuffer.pos <= context->decompressedBuffer.size );
		
		context->decompressedBuffer.data[context->decompressedBuffer.pos++] = invoer;
	}
	
	return( 0 );
}

	int
decompress_getc_buffer(void *pUserValue)
{
	PGPsdaDecryptionContext	*context = (PGPsdaDecryptionContext *) pUserValue;
	int						result;
	
	pgpAssert( context->decryptedBuffer.pos <= context->decryptedBuffer.used );
	
	if( IsntErr( context->decompressionErr ) && context->remainingDecompressBytes != 0 )
	{
		result = context->decryptedBuffer.data[context->decryptedBuffer.pos++];
		
		context->remainingDecompressBytes--;
		
		if( context->remainingDecompressBytes != 0 &&
			context->decryptedBuffer.pos == context->decryptedBuffer.used )
		{
			context->decompressionErr = FillDecryptedBuffer( context );
			if( IsErr( context->decompressionErr ) )
			{
				result = EOF;
			}
		}
	}
	else
	{
		result = EOF;
	}
	
	return( result );
}

	static OSStatus
DecryptFileFork(
	PGPsdaDecryptionContext *context,
	short					fileRef,
	PGPUInt32				forkLength)
{
	OSStatus	err				= noErr;
	PGPUInt32	bytesRemaining 	= forkLength;

	while( bytesRemaining != 0 && IsntErr( err ) )
	{
		PGPUInt32	bytesToRead;
		
		bytesToRead = context->outputBuffer.size;
		if( bytesToRead > bytesRemaining )
			bytesToRead = bytesRemaining;
			
		err = ReadFromDecompressor( context, context->outputBuffer.data, bytesToRead );
		if( IsntErr( err ) )
		{
			long	count = bytesToRead;

			err = FSWrite( fileRef, &count, context->outputBuffer.data );
		}
		
		bytesRemaining -= bytesToRead;
	}
	
	return( err );
}

	static OSStatus
DecryptFile(
	PGPsdaDecryptionContext *context,
	PGPBoolean				isRootFile)
{
	OSStatus			err;
	PGPsdaFileHeader	header;

	err = ReadFromDecompressor( context, &header, sizeof( header ) );
	if( IsntErr( err ) )
	{
		FInfo	finderInfo;
		FSSpec	fileSpec;
		
		finderInfo.fdType 		= header.fileType;
		finderInfo.fdCreator 	= header.fileCreator;
		finderInfo.fdFlags 		= header.finderFlags;
		finderInfo.fdFldr		= header.windowID;
		
		/* Clear the inited bit so the Finder will read the bundle */
		if( ( finderInfo.fdFlags & kHasBundle ) != 0 )
			finderInfo.fdFlags &= ~kHasBeenInited;
			
		if( isRootFile )
		{
			/* Do not restore icon location. Use "magic placement" instead */

			finderInfo.fdLocation.v	= -1;
			finderInfo.fdLocation.h	= -1;
		}
		else
		{
			finderInfo.fdLocation = header.windowIconPos;
		}
		
		fileSpec.vRefNum 	= context->outFolderSpec->vRefNum;
		fileSpec.parID 		= context->outDirID;
		
		CopyPString( header.name, fileSpec.name );
		
		err = FSpCreate( &fileSpec, finderInfo.fdCreator, finderInfo.fdType,
					header.scriptCode );
		if( IsntErr( err ) )
		{
			if( header.dataForkLength != 0 )
			{
				short	fileRef;
				
				err = FSpOpenDF( &fileSpec, fsRdWrPerm, &fileRef );
				if( IsntErr( err ) )
				{
					err = DecryptFileFork( context, fileRef, header.dataForkLength );
					
					FSClose( fileRef );
				}
			}
			
			if( IsntErr( err ) && header.resForkLength != 0 )
			{
				short	fileRef;
				
				err = FSpOpenRF( &fileSpec, fsRdWrPerm, &fileRef );
				if( IsntErr( err ) )
				{
					err = DecryptFileFork( context, fileRef, header.resForkLength );
					
					FSClose( fileRef );
				}
			}
			
			if( IsntErr( err ) )
			{
				CInfoPBRec	cpb;
				
				err = FSpGetCatInfo( &fileSpec, &cpb );
				if( IsntErr( err ) )
				{
					cpb.hFileInfo.ioFlFndrInfo = finderInfo;
					
					cpb.hFileInfo.ioFlCrDat = header.creationDate;	
					cpb.hFileInfo.ioFlMdDat = header.modificationDate;		

					cpb.hFileInfo.ioFlXFndrInfo.fdXFlags = header.extendedFinderFlags;
					cpb.hFileInfo.ioFlXFndrInfo.fdScript = header.scriptCode;
								
					err = FSpSetCatInfo( &fileSpec, &cpb );
				}
			}
			
			if( IsntErr( err ) )
			{
				if( header.locked )
				{
					(void) FSpSetFLock( &fileSpec );
				}
			}
			else
			{
				FSpDelete( &fileSpec );
			}
		}
	}
	
	return( err );
}

	static OSStatus
DecryptFolder(
	PGPsdaDecryptionContext *context,
	PGPBoolean				isRootFolder)
{
	OSStatus				err;
	PGPsdaBeginFolderHeader	header;
	
	err = ReadFromDecompressor( context, &header, sizeof( header ) );
	if( IsntErr( err ) )
	{
		FSSpec	folderSpec;
		
		folderSpec.vRefNum 	= context->outFolderSpec->vRefNum;
		folderSpec.parID 	= context->outDirID;
		
		CopyPString( header.name, folderSpec.name );
		
		err = FSpDirCreate( &folderSpec, header.scriptCode, &context->outDirID );
		if( IsntErr( err ) )
		{
			CInfoPBRec	cpb;
			
			err = FSpGetCatInfo( &folderSpec, &cpb );
			if( IsntErr( err ) )
			{
				cpb.dirInfo.ioDrCrDat 				= header.creationDate;
				cpb.dirInfo.ioDrUsrWds 				= header.folderInfo;
				cpb.dirInfo.ioDrFndrInfo.frScript	= header.scriptCode;
				cpb.dirInfo.ioDrFndrInfo.frXFlags	= header.extendedFlags;
			
				if( isRootFolder )
				{
					/* Magic icon placement for root items */
					
					cpb.dirInfo.ioDrUsrWds.frLocation.v	= -1;
					cpb.dirInfo.ioDrUsrWds.frLocation.h	= -1;
				}
				
				err = FSpSetCatInfo( &folderSpec, &cpb );
			}
		}
	}
	
	return( err );
}

	static OSStatus
FinishDecryptFolder(PGPsdaDecryptionContext *context)
{
	OSStatus				err;
	PGPsdaEndFolderHeader	header;
	
	err = ReadFromDecompressor( context, &header, sizeof( header ) );
	if( IsntErr( err ) )
	{
		CInfoPBRec	cpb;
		Str32		folderName;
		
		pgpClearMemory( &cpb, sizeof( cpb ) );
		
		cpb.dirInfo.ioVRefNum	= context->outFolderSpec->vRefNum;
		cpb.dirInfo.ioDrDirID	= context->outDirID;
		cpb.dirInfo.ioNamePtr	= folderName;
		cpb.dirInfo.ioFDirIndex	= -1;
		
		err = PBGetCatInfoSync( &cpb );
		if( IsntErr( err ) )
		{
			cpb.dirInfo.ioDrDirID				= cpb.dirInfo.ioDrParID;
			cpb.dirInfo.ioDrMdDat 				= header.modificationDate;
			cpb.dirInfo.ioDrFndrInfo.frScroll	= header.windowScroll;
			
			err = PBSetCatInfoSync( &cpb );
		}
		
		context->outDirID = cpb.dirInfo.ioDrParID;
	}
	
	return( err );
}

	static OSStatus
DecryptArchiveItems(PGPsdaDecryptionContext *context)
{
	OSStatus	err 			= noErr;
	PGPBoolean	processingItems = TRUE;
	
	while( processingItems && IsntErr( err ) )
	{
		PGPsdaHeaderType	headerType;
		
		err = ReadFromDecompressor( context, &headerType, sizeof( headerType ) );
		if( IsntErr( err ) )
		{
			PGPUInt32	folderDepth = 0;
			
			switch( headerType )
			{
				case kPGPsdaHeaderType_File:
					err = DecryptFile( context, folderDepth == 0 );
					break;
					
				case kPGPsdaHeaderType_BeginFolder:
					err = DecryptFolder( context, folderDepth == 0 );
					++folderDepth;
					break;
					
				case kPGPsdaHeaderType_EndFolder:
					err = FinishDecryptFolder( context );
					--folderDepth;
					break;
					
				case kPGPsdaHeaderType_EndArchive:
					processingItems = FALSE;
					break;
					
				default:
					pgpDebugMsg( "\pDecryptArchiveItems(): Invalid header type" );
					err = kSDAError_InvalidArchive;
					break;
			}
		}
	}
	
	return( err );
}

	static OSStatus
DecryptArchive(PGPsdaDecryptionContext *context)
{
	OSStatus	err = noErr;
	CInfoPBRec	cpb;
	
	/* Delete output spec if it exists */
	if( FSpGetCatInfo( context->outFolderSpec, &cpb ) == noErr )
	{
		err = FSpDelete( context->outFolderSpec );
	}
	
	if( IsntErr( err ) )
	{
		err = FSpDirCreate( context->outFolderSpec, smSystemScript,
					&context->outDirID );
		if( IsntErr( err ) )
		{
			(void) FSpGetCatInfo( context->outFolderSpec, &cpb );
				cpb.dirInfo.ioDrUsrWds.frLocation.h 	= -1;	// Magic icon placement
				cpb.dirInfo.ioDrUsrWds.frLocation.v 	= -1;	// Magic icon placement
			(void) FSpSetCatInfo( context->outFolderSpec, &cpb );

			err = DecryptArchiveItems( context );
			if( IsntErr( err ) )
			{
				SHA::Digest	digest;
				
				context->hash->Final( &digest );
				
				if( ! pgpMemoryEqual( &digest, context->archiveHeader.encryptedDataHash,
							sizeof( context->archiveHeader.encryptedDataHash ) ) )
				{
					err = kSDAError_ChecksumFailed;
				}
			}
			else
			{
				/* Try to delete possibly empty output folder */
				(void) FSpDelete( context->outFolderSpec );
			}
		}
	}
	
	return( err );
}

	OSStatus
PGPsdaDecrypt(
	const FSSpec 				*sourceFileSpec,
	const FSSpec 				*outFolderSpec,
	const char					*passphrase,
	PGPsdaDecryptProgressProc 	progressProc,
	PGPUserValue 				userValue)
{
	OSStatus					err = noErr;
	PGPsdaDecryptionContext		sdaContext;
	
	pgpAssert( IsntNull( sourceFileSpec ) );
	pgpAssert( IsntNull( outFolderSpec ) );
	
	/* Lock down the context because it contains sensitive items */
	HoldBuffer( &sdaContext, sizeof( sdaContext ) );
	
	pgpClearMemory( &sdaContext, sizeof( sdaContext ) );

	sdaContext.sourceFileSpec	= sourceFileSpec;
	sdaContext.outFolderSpec	= outFolderSpec;
	sdaContext.progressProc		= progressProc;
	sdaContext.userValue		= userValue;
	
	if( IsntNull( passphrase ) )
	{
		CopyCString( passphrase, sdaContext.passphrase );
	}
	else
	{
		err = PGPsdaDecryptPassphraseDialog( sdaContext.passphrase );
	}
	
	if( IsntErr( err ) )
	{
		err = BeginSDADecryption( &sdaContext );
		if( IsntErr( err ) )
		{
			err = DecryptArchive( &sdaContext );
			
			EndSDADecryption( &sdaContext );
		}
	}
	
	UnholdBuffer( &sdaContext, sizeof( sdaContext ), TRUE );

	return( err );
}

	static pascal Boolean
SDAFileFilterProc(CInfoPBRec *cpb)
{
	Boolean	shouldFilter;
	
	if( cpbIsFile( cpb ) )
	{
		if( cpbFileType( cpb ) == 'APPL' &&
			( cpbFileCreator( cpb ) == kPGPsdaCreator ||
			  cpbFileCreator( cpb ) == kPGPseaCreator ) )
		{
			shouldFilter = FALSE;
		}
		else
		{
			shouldFilter = TRUE;
		}
	}
	else
	{
		shouldFilter = FALSE;
	}
	
	return( shouldFilter );
}

	static OSStatus
ChooseSDA(FSSpec *sdaFileSpec)
{
	StandardFileReply	sfReply;
	SFTypeList			typeList;
	FileFilterUPP		filterUPP = NewFileFilterProc( SDAFileFilterProc );
	
	typeList[0] = 'APPL';
	
	StandardGetFile( filterUPP, 1, typeList, &sfReply );
	*sdaFileSpec = sfReply.sfFile;
	
	DisposeRoutineDescriptor( filterUPP );
	
	return( sfReply.sfGood ? noErr : userCanceledErr );
}

	static void
UpdateProgressDialog(DialogPtr progressDialog)
{
	if( ! EmptyRgn( ((WindowPeek) progressDialog)->updateRgn ) )
	{
		BeginUpdate( progressDialog );
			UpdateDialog( progressDialog, progressDialog->visRgn );
		EndUpdate( progressDialog );
	}
}

	static void
SetProgressBarPctComplete(
	DialogPtr 	progressDialog,
	PGPUInt32	pctComplete)
{
	DialogItemType	itemType;
	Rect			itemRect;

	if( HaveAppearanceMgr() )
	{
		ControlHandle	progressBar;
		
		GetDialogItem( progressDialog, kProgressBarDialogItemID,
				&itemType, (Handle *) &progressBar, &itemRect );
		pgpAssert( IsntNull( progressBar ) );
		
		SetControlValue( progressBar, pctComplete );
	}
}

	static OSStatus
DecryptProgressProc(
	const PGPsdaDecryptProgressInfo 	*progressInfo,
	PGPUserValue 						userValue)
{
	OSStatus			err = noErr;
	PGPUInt32			ticks;
	static PGPUInt32	sLastProgressTicks = 0;
	
	/* Throttle the progress events to 4/second */
	
	ticks = TickCount();
	if( ticks - sLastProgressTicks >= 15 )
	{
		DialogPtr	progressDialog = (DialogPtr) userValue;
		
		sLastProgressTicks = ticks;
		
		if( IsntNull( progressDialog ) )
		{
			GrafPtr	savePort;
			UInt32	pctComplete;
			
			GetPort( &savePort );
			SetPort( progressDialog );
			
			if( progressInfo->totalBytesToProcess > 0x02000000 )
			{
				pctComplete = ( (double) progressInfo->bytesProcessed * (double) 100.0 ) /
								(double) progressInfo->totalBytesToProcess;
			}
			else
			{
				pctComplete = ( progressInfo->bytesProcessed * 100 ) /
								progressInfo->totalBytesToProcess;
			}
			
			if( IsWindowVisible( progressDialog ) )
			{
				EventRecord		theEvent;
				
				/* Redraw trhe dialog if necessary */
				UpdateProgressDialog( progressDialog );
				
				/* Update the progress bar */
				SetProgressBarPctComplete( progressDialog, pctComplete );
				
				/* Check for cancel */
				
				if( GetNextEvent( mDownMask | mUpMask | keyDownMask | keyUpMask,
						&theEvent ) )
				{
					switch( theEvent.what )
					{
						case keyDown:
						{
							if( ( theEvent.message & charCodeMask ) == char_Escape &&
		 						( theEvent.message & keyCodeMask )  == vkey_Escape )
		 					{
		 						err = userCanceledErr;
		 					}
		 					else if( IsCmdChar( &theEvent, char_Period ) )
		 					{
		 						err = userCanceledErr;
							}
							
							if( IsErr( err ) )
							{
								UInt32			finalDelay;
								ControlHandle	stopButton;
								DialogItemType	itemType;
								Rect			itemRect;
								
								GetDialogItem( progressDialog, kStopButtonID,
										&itemType, (Handle *) &stopButton, &itemRect );
								pgpAssert( IsntNull( stopButton ) );

								HiliteControl( stopButton, kControlButtonPart );
								Delay( 8, &finalDelay );
								HiliteControl( stopButton, 0 );
							}
						}
						
						case mouseDown:
						{
							Point			mouseLoc;
							ControlHandle	theButton;
							
							mouseLoc = theEvent.where;
							GlobalToLocal( &mouseLoc );
						
							if( FindControl( mouseLoc, progressDialog, &theButton ) ==
										kControlButtonPart )
							{
								if( TrackControl( theButton, mouseLoc, NULL ) != 0 )
								{
									err = userCanceledErr;
								}
							}
						}
						
						default:
							break;
					}
				}
			}
			else
			{
				PGPUInt32	startTicks;
				
				/*
				** We show the progress window after 2 seconds only if
				** we are less than 50% complete. The starting ticks are stored
				** in the refCon of the dialog.
				*/
				
				startTicks = GetWRefCon( progressDialog );
				
				if( TickCount() - startTicks >= 120 && pctComplete <= 50 )
				{
					SetProgressBarPctComplete( progressDialog, pctComplete );

					ShowWindow( progressDialog );
					SelectWindow( progressDialog );
					
					UpdateProgressDialog( progressDialog );
				}
			}
			
			SetPort( savePort );
		}
	}
	
	return( err );
}

	static void
ReportError(OSStatus err)
{
	if( IsErr( err ) && err != userCanceledErr )
	{
		Str255	msg;
		
		switch( err )
		{
			case kSDAError_ChecksumFailed:
			{
				GetIndString( msg, kSDAStringListResID,
						kIntegrityCheckFailureStrIndex );
				break;
			}
			
			case kSDAError_InvalidArchive:
			{
				GetIndString( msg, kSDAStringListResID,
						kCorruptedArchiveStrIndex );
				break;
			}

			default:
			{
				Str255	errorStr;
				
				GetIndString( msg, kSDAStringListResID,
						kErrorOccurredStrIndex );
				GetOSErrorString( err, errorStr );
				PrintPString( msg, msg, errorStr );
				break;
			}
		}
		
		SysBeep( 1 );
		ParamText( msg, "\p", "\p", "\p" );
		
		StopAlert( kSDAErrorAlertResID, NULL );
	}
}

	void
DecryptSDA(Boolean promptForSource)
{
	FSSpec		sourceSpec;
	FSSpec		destSpec;
	OSStatus	err;
	UInt16		modifiers;
	DialogPtr	progressDialog = NULL;
	
	modifiers = GetModifiers();
	
	if( modifiers == ( controlKey | optionKey | cmdKey ) || promptForSource )
	{
		/*
		** Special "hidden" compination to prompt for a source archive.
		** This is useful if the resource fork of one archive is corrupted.
		** The user can use a different archive to decrypt the corrupted one.
		*/
		
		err = ChooseSDA( &sourceSpec );
	}
	else
	{
		err = GetSpecFromRefNum( CurResFile(), &sourceSpec );
	}
	
	if( IsntErr( err ) )
	{
		Str32	suffix;
		
		/* Determine default output spec */

		destSpec = sourceSpec;

		GetIndString( suffix, kSDAStringListResID, kNameExtensionStrIndex );
		
		if( PStringHasSuffix( destSpec.name, suffix, FALSE ) )
		{
			destSpec.name[0] -= suffix[0];
		}
	}
	
	if( IsntErr( err ) )
	{
		Boolean	promptForOutput = FALSE;
		
		if( modifiers == optionKey )
		{
			promptForOutput = TRUE;
		}
		else
		{
			HParamBlockRec	pb;
			Str32			volumeName;
			
			pgpClearMemory( &pb, sizeof( pb ) );
			
			pb.volumeParam.ioVRefNum = destSpec.vRefNum;
			pb.volumeParam.ioNamePtr = volumeName;
			
			err = PBHGetVInfoSync( &pb );
			if( IsntErr( err ) )
			{
				promptForOutput = ( pb.volumeParam.ioVAtrb & kVolumeAttributeLockMask ) != 0;
			}
		}
		
		if( IsntErr( err ) )
		{
			if( promptForOutput )
			{
				Str255				prompt;
				StandardFileReply	sfReply;
				
				GetIndString( prompt, kSDAStringListResID, kDecryptToPromptStrIndex );
				
				StandardPutFile( prompt, destSpec.name, &sfReply );
				if( sfReply.sfGood )
				{
					destSpec = sfReply.sfFile;
				}
				else
				{
					err = userCanceledErr;
				}
			}
			else
			{
				err = FSpGetUniqueSpec( &destSpec, &destSpec );
			}
		}
	}

	if( IsntErr( err ) )
	{
		short	resID;
		
		/* Set up progress dialog */
		
		if( HaveAppearanceMgr() )
		{
			resID = kSDAAppearanceProgressDialogResID;
		}
		else
		{
			resID = 0;
		}
		
		progressDialog = GetNewDialog( resID, NULL, (WindowPtr) -1L );
		if( IsNull( progressDialog ) )
		{
			err = memFullErr;
		}
	}
	
	if( IsntErr( err ) )
	{
		Boolean	tryEmptyPassphrase = TRUE;
			
		while( TRUE )
		{
			SetWRefCon( progressDialog, TickCount() );
			
			err = PGPsdaDecrypt( &sourceSpec, &destSpec,
						tryEmptyPassphrase ? "" : NULL, DecryptProgressProc,
						progressDialog );
			
			HideWindow( progressDialog );
			
			if( err == kSDAError_IncorrectPassphrase )
			{
				if( tryEmptyPassphrase )
				{
					tryEmptyPassphrase = FALSE;
				}
				else
				{
					SysBeep( 1 );
				}
			}
			else
			{
				break;
			}
		}

		DisposeDialog( progressDialog );
	}
	
	if( IsErr( err ) && err != userCanceledErr )
	{
		ReportError( err );
	}
}
