/*____________________________________________________________________________
	PGPPlug.cp
	
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: PGPPlug.cp,v 1.19 1999/05/23 16:51:30 heller Exp $
____________________________________________________________________________*/
#include <Icons.h>
#include <TextUtils.h>

#include "MacStrings.h"
#include "MacIcons.h"
#include "MacFiles.h"

#include "PGPPlug.h"
#include "CTranslator.h"
#include "CDecryptTranslator.h"
#include "TranslatorFactory.h"
#include "PassphraseCache.h"
#include "TranslatorPrefs.h"
#include "PluginProcInfo.h"
#include "TranslatorStrings.h"
#include "TranslatorUtils.h"
#include "WarningAlert.h"
#include "pgpMacCustomContextAlloc.h"
#include "pgpClientLib.h"

#if PGP_DEMO
#include "PGPDemo.h"
#endif

// same as used in the BNDL resource
const SInt16	kAboutPGPPluginIconID				=	128;

// obtained 3/17/97, also obtained 881 thru 884
const short		kModuleID	 = 880;



#pragma export on

// these routines are called by the plugin code resource
RoutineDescriptor	PlugInInitRD		=
	BUILD_ROUTINE_DESCRIPTOR( uppPlugInInitProcInfo, PlugInInit );
RoutineDescriptor	PlugInFinishRD		=
	BUILD_ROUTINE_DESCRIPTOR( uppPluginFinishProcInfo, PlugInFinish );
RoutineDescriptor	PlugInConfigRD		=
	BUILD_ROUTINE_DESCRIPTOR( uppPluginConfigProcInfo, PlugInConfig );
RoutineDescriptor	GetTranslatorInfoRD =
	BUILD_ROUTINE_DESCRIPTOR( uppGetTranslatorInfoProcInfo, GetTranslatorInfo );
RoutineDescriptor	CanTranslateRD	=
	BUILD_ROUTINE_DESCRIPTOR( uppCanTranslateProcInfo, CanTranslate );
RoutineDescriptor	TranslateFileRD		=
	BUILD_ROUTINE_DESCRIPTOR( uppTranslateFileProcInfo, TranslateFile );

#pragma export off




static Boolean			sInited			= false;
static PGPContextRef	sContext		= kInvalidPGPContextRef;
static PGPtlsContextRef	sTLSContext		= kInvalidPGPtlsContextRef;

	static CComboError
InitMyWorld( void )
{
	CComboError	err;
	
	if ( ! sInited )
	{
		err.pgpErr = pgpNewContextCustomMacAllocators( &sContext );
		if ( err.IsntError() )
		{
			err.pgpErr = PGPNewTLSContext( sContext, &sTLSContext );
			if ( err.IsntError() )
			{
				InitPassphraseCaches( sContext );
				
		#if PGP_BUSINESS_SECURITY
					err.pgpErr = PGPCheckAutoUpdateKeysFromServer( 
										PGPGetContextMemoryMgr( sContext ), TRUE,
										NULL, NULL );
		#endif
			}
		}
		
		sInited	= true;	
	}
	
	return( err );
}

	static void
DisposeMyWorld( void )
{
	if ( sInited )
	{
		if( PGPtlsContextRefIsValid( sTLSContext ) )
		{
			PGPFreeTLSContext( sTLSContext );
			sTLSContext = kInvalidPGPtlsContextRef;
		}
		
		if( PGPContextRefIsValid( sContext ) )
		{
			PGPFreeContext( sContext );
			sContext = kInvalidPGPContextRef;
		}
		
		// nothing yet
		sInited	= false;
	}
	
	DisposePassphraseCaches();
}

#pragma mark -



	pascal PluginError
PlugInInit(
	emsUserGlobals 	**globalsHandle,
	short 			eudAPIVersion,
	emsMailConfigP	mailConfig,
    emsPluginInfoP 	pluginInfo)
{
	const uchar *sModuleDescription = "\pEudora PGP Plugin";
	CComboError	err;  
	PluginError	eudoraErr	= EMSR_OK;
	
	(void) globalsHandle;
	(void) eudAPIVersion;
	(void) mailConfig;
	
	pgpLeaksBeginSession( "main" );
	
	err	= InitMyWorld();
	if( err.IsntError() )
	{
		Handle	tempSuite;
		
		pgpClearMemory( pluginInfo, sizeof( *pluginInfo ) );
		
		pluginInfo->size		= sizeof( *pluginInfo );
		pluginInfo->id			= kModuleID;
		pluginInfo->numTrans	= kPGPNumTranslators;
		
		MacLeaks_Suspend();	// Eudora keeps the stuff we're about to allocate
		
		PtrToHand( &sModuleDescription[ 0 ], (Handle *) &pluginInfo->desc,
						sModuleDescription[ 0 ] + 1 );
						
		err.err = GetIconSuite( &tempSuite, kAboutPGPPluginIconID,
						svAllAvailableData);
		if( err.IsntError() && IsntNull( tempSuite ) )
		{
			err.err = DuplicateIconSuite( tempSuite, &pluginInfo->icon );
			
			DisposeIconSuite( tempSuite, true );
		}
		
		MacLeaks_Resume();	// Eudora keeps the stuff we just allocated
	}
	
	if ( err.IsError( ) )
	{
		pgpDebugPStr( "\pfailure in PlugInInit" );
		eudoraErr	= EMSR_UNKNOWN_FAIL;
	}
	else
	{
		eudoraErr = EMSR_OK;
	}
	
#if BETA
	static Boolean	sNotified	= false;
	if ( BetaExpired() && ! sNotified )
	{
	sNotified	= true;
	WarningAlert( kWANoteAlertType, kWAOKStyle,
		"\pThis BETA version of PGP Eudora Plugin has expired."
		"You may decrypt and verify, but not encrypt or sign.  "
		"Please obtain the latest version from the PGP web site." );
	}
#endif

#if PGP_DEMO
	if( err.IsntError() )
	{
		if( ShowDemoDialog( FALSE ) == kDemoStateExpired )
		{
			eudoraErr = EMSR_UNKNOWN_FAIL;
		}
	}
#endif

	if( ! PGPClientVerifyEnvironment() )
	{
		eudoraErr = EMSR_UNKNOWN_FAIL;
	}
	
	return( eudoraErr );
}



	pascal PluginError
PlugInFinish(emsUserGlobals **globalsHandle)
{
	(void) globalsHandle;
	
	DisposeMyWorld( );
	
	// pgpLeaksEndSession();
	
	return( EMSR_OK );
}




	pascal PluginError
GetTranslatorInfo(
	emsUserGlobals **globalsHandle,
	emsTranslatorP transInfo)
{
	CComboError		err;
	PluginError		eudoraErr	= EMSR_OK;
	
	(void) globalsHandle;
	
	err	= InitMyWorld();
	if ( err.IsntError() )
	{
		// Make sure that we are given a valid ID
		if( transInfo->id >= kFirstTranslatorID &&
			transInfo->id < kLastTranslatorIDPlusOne )
		{
			eudoraErr = CTranslator::GetInfo( transInfo );
		}
		else
		{
			pgpDebugPStr( "\pillegal translator ID" );
		}
	}
	
	if ( err.IsError() )
		eudoraErr = CComboErrorToEudoraError( err );
		
	return( eudoraErr );
}

	pascal PluginError
PlugInConfig(
	emsUserGlobals 	**globalsHandle,
	emsMailConfigP	mailConfig)
{
	PGPError		err;
	PluginError		eudoraErr = EMSR_OK;
	
	(void) globalsHandle;
	(void) mailConfig;
	
	err = PGPPreferencesDialog( sContext, kInvalidPGPKeySetRef,
				kInvalidPGPPrefRef, kInvalidPGPPrefRef, NULL );
	if( IsntPGPError( err ) )
	{
		RefreshPrefs();
	}
	
	return( eudoraErr );
}

	pascal PluginError
CanTranslate(
	emsUserGlobals 		**globalsHandle,
	emsTranslatorP 		trans,
	emsDataFileP 		inTransData,
    emsResultStatusP 	transStatus)
{
	CComboError	err;
	PluginError	eudoraErr	= EMSR_OK;
	
	(void) globalsHandle;
	
	transStatus->size	= sizeof( *transStatus );
	transStatus->desc 	= NULL;
	transStatus->error 	= NULL;
	transStatus->code 	= EMSR_OK;

	err	= InitMyWorld( );
	if ( err.IsntError() )
	{
		CTranslator *	translator	= nil;
		
		translator = CreateTranslator( NULL, NULL, NULL, (TranslatorID) trans->id );
		if ( IsntNull( translator ) )
		{
			eudoraErr = translator->CanTranslate( trans, inTransData, transStatus );
		
			delete translator;
		}
		else
		{
			eudoraErr = EMSR_INVALID_TRANS;
		}
	}
	
	if ( err.IsError() )
		eudoraErr = CComboErrorToEudoraError( err );
		
	return( eudoraErr );
}

	pascal PluginError
TranslateFile(
	emsUserGlobals 		**globalsHandle,
	emsTranslatorP 		trans,
	emsDataFileP 		inFile,
    EMSProgressUPP 		progress,
    emsDataFileP 		outFile,
    emsResultStatusP 	transStatus)
{
	CInfoPBRec		cpb;
	ulong			secs;
	CComboError		err;
	PluginError		eudoraErr	= EMSR_OK;

	(void) globalsHandle;
	
	transStatus->size	= sizeof( *transStatus );
	transStatus->desc 	= NULL;
	transStatus->error 	= NULL;
	transStatus->code 	= EMSR_OK;

	outFile->mimeInfo = NULL;
	
	(void) FSpGetCatInfo( &inFile->file, &cpb );	// info needed below

	// convenient to do this here, globally
	secs	= PrefGetSigningPassphraseCacheSeconds( );
	gSigningPassphraseCache->SetCacheSeconds( secs );
	
	secs	= PrefGetEncryptionPassphraseCacheSeconds( );
	gDecryptionPassphraseCache->SetCacheSeconds( secs );
	
	err	= InitMyWorld( );
	if ( err.IsntError() )
	{
		const ulong		kMinHeapSpace	= 300 * 1024UL;
		
		if ( FreeMem() < kMinHeapSpace )
		{
			Str255	msg;
			
			GetIndString( msg,
				kTranslatorErrorStringsResID, kOutOfMemoryStrIndex );
			WarningAlert( kWANoteAlertType, kWAOKStyle, msg );
			eudoraErr			= CComboErrorToEudoraError( err );
			transStatus->code	= EMSR_TRANS_FAILED;
		}
	}
	
	
	if ( err.IsntError() )
	{
		CTranslator *	translator	= nil;
		
		translator = CreateTranslator( sContext, sTLSContext,
					progress, (TranslatorID)trans->id );
		if ( IsntNull( translator ) )
		{
			err	= translator->PrepareToTranslate();
			if ( err.IsntError() )
			{
				eudoraErr = translator->TranslateFile( trans, inFile,
									outFile, transStatus);
			}
			
			delete translator;
		}
		else
		{
			eudoraErr = EMSR_INVALID_TRANS;
		}
	}
	
	// Eudora doesn't like it if there is no output file, even if
	// we're returning an error.
	if ( ! FSpExists( &outFile->file ) )
	{
		(void) FSpCreate( &outFile->file, cpbFileCreator( &cpb ), cpbFileType( &cpb ),
				smSystemScript );
	}

	if ( err.IsError() )
	{
		eudoraErr = CComboErrorToEudoraError( err );
	}
	
	return eudoraErr;
}
