/* smplpgp.h : Declares interfaces for the PGP for Windows 4.0 Simple DLL base functions. */

/*	Confidential to PGP.	*/
/*	Copyright (c) 1995 PGP.	*/

/*******************************************************************/
/*
                         Simple PGP API
                         
    These functions are inteded to be used by most applications
    that require simple use of the PGP DLL's services.  Most of
    these functions correspond to running the MS-DOS or Unix
    PGP with the "+Batchmode" flag turned on.  They execute,
    and if any error is encountered they stop and return an error
    code.  They do not do any kind of user interaction.
*/

/**************************************************************************

Simplified key selection function:

SimplePGPKeySel -- This function is intended to be a simplified
version of KeySel.  It is primarily targeted to Visual Basic 
developers and others who cannot support an interface requiring
the use of callbacks


Arguments:

HWND hWnd       - Window handle of calling application

LPCSTR Caption  - Pointer to a null-terminated string.  This caption
                  will appear in the key selection dialog box's
                  title bar
                  
BOOL AllowMultSel - if TRUE, allow selection of multiple user IDs
                  in the list.                  

int KeyRingType - A constant specifying the type keyring(s) (public or
                  private) to be searched.  The constants are:
                  
                  KEYSEL_PUBLIC Public keyring(s)
                  KEYSEL_SECRET Private Keyring(s)
                  KEYSEL_GROUP Define groups of public keys
                  
LPSTR SelectedtList - Pointer to a buffer that this function will fill
                with the selected UserIDs/KeyIDs.  Depending upon the
                value of IncludeWhat, the buffer will contain only UserIDs,
                Only KeyIDs, or both.
                
                If IncludeWhat is set to INCLUDE_ONLYUSERIDS, then
                the buffer will be filled with user IDs, seperated by
                newline ('\n') characters.  The last userID willbe terminated
                with a newline followed by a null.
                
                If IncludeWhat is set to INCLUDE_ONLYKEYIDS, then
                the buffer will be filled with KeyIDs, seperated by
                newline ('\n') characters.  The last KeyID willbe terminated
                with a newline followed by a null.
                
                If IncludeWhat is set to INCLUDE_BOTH, then
                the buffer will be filled with user ID/KeyID pairs like so:
                
                UserID\nKeyID\nUserID\nKeyID\n . . .
                
                The last KeyID willbe terminated with a newline 
                followed by a null.
                
                If the buffer is not large enough to hold all selections,
                The user will be informed and wil be given another chance
                to make selections. 

int SelectedListLength -- The caller must initialize this argument to
                the length of the buffer specified by SelectedList
                
int FAR *SelectedCount -- Upon successful return, this argument will
                reflect the number of items selected by the user.

int IncludeWhat -- A flag that indicates if the caller wants the 
                SelectedList buffer to contain UserIDs, KeyIDs, or both:
                
                INCLUDE_ONLYUSERIDS Specifies that only userIDs are
                                    to be plcaed in the SelectedList
                                    buffer
                                    
                INCLUDE_ONLYKEYIDS  Specifies that only KeyIDs are
                                    to be placed in the buffer.
                                    
                INCLUDE_BOTH        Specifies that both UserIDs and
                                    KeyIDs are to be placed in the                     
                                    SelectedList buffer
                                    
                We recommend that this argument be set to INCLUDE_ONLYKEYIDS
                for two reasons.  First, KeyIDs are generally shorter,
                so more of them can fit in a buffer.  Second, Two keys
                may well have the same UserID, but the KeyID is unique
                to each key.
                
                                                                                                   
                
LPSTR UidKid -  Pointer to a null-terminated character string containing
                a user ID fragment or a keyID to search for.  Only keys
                with uids or keyids matching UidKid will appear in the
                Dialog's list box.  A null string matches all.  The
                search is case-insensitive.
                
int Keytype   - A bitmap that specifies the type of keys to display:

                 KEYS_OLD  01  Tradional PGP RSA Keys
                 KEYS_ENCR 02  New style PGP Encryption-only keys
                 KEYS_SIGN 04  New style PGP Signature-only keys
                
                
LPSTR CAKey  -  Pointer to a null-terminated string containing the
                ASCII representation of the least significant three
                bytes of a 'Certification Authority'  only keys that
                have been certified (signed) by CAKey will be displayed
                in the dialog's list box.  A null string or a null
                pointer matches all

BOOL ShowKeyDetails - If TRUE, then the dialog box starts up in its 'wide'
                form, displaying not only the user IDs, but also the
                KeyID, length, and creation date.
                
                If FALSE, then the dialog box starts up in its 'narrow'
                form, displaying only user IDs.
                
                A button in the dialog box allows the user to switch
                between the two forms.                               
                
UINT DisplayType -
				Flag to display groups, expired keys, disabled keys, or revoked keys
				The flag bit values are:
					KEYSEL_SHOW_GROUPS
					KEYSEL_SHOW_EXP
					KEYSEL_SHOW_DIS
					KEYSEL_SHOW_REV

UINT SharedButtonID - 
				One button position is shared between buttons identified by these ID's:
					KEYSEL_GROUP_DEF
					plus a code to disable all the buttons:
					KEYSEL_DISABLE_SHARED_BUTTONS
                
                
*/                

/* Keyring Types: */
#define KEYSEL_PUBLIC 1
#define KEYSEL_SECRET 2
#define KEYSEL_GROUP 3

// Show or Hide goups or expired, disabled, or revoked keys
#define KEYSEL_SHOW_GROUPS  0x0100
#define KEYSEL_SHOW_EXP  		  0x0200
#define KEYSEL_SHOW_DIS  		   0x0400
#define KEYSEL_SHOW_REV  		  0x0800

// Identify the shared button to use
#define KEYSEL_GROUP_DEF								1
#define KEYSEL_DISABLE_SHARED_BUTTONS    4

#define KEYS_OLD  01  //Tradional PGP RSA Keys
#define KEYS_ENCR 02  //New style PGP Encryption-only keys
#define KEYS_SIGN 04  //New style PGP Signature-only keys

/* Note: The key type values below are copied 
			from PGP.h  and used in the kernel. */
/*	Public key encryption algorithm selector bytes. */
//#define RSA_ALGORITHM_BYTE	1	/*	use RSA	*/
//#define	RSA_ENC_ONLY_BYTE	2
//#define	RSA_SIG_ONLY_BYTE	3
//#define	DSS_SIG_ONLY_BYTE	6


#define INCLUDE_ONLYUSERIDS 1
#define INCLUDE_ONLYKEYIDS  2
#define INCLUDE_BOTH        3

int __declspec(dllexport) FAR PASCAL SimplePGPKeySel(
											HWND hWnd,
											LPSTR Caption,
											BOOL AllowMultSel,
											int KeyRingType, 
											LPSTR SelectedList,
											int SelectedListLength,
											int FAR *SelectedCount,
											int IncludeWhat,
											LPSTR UidKid, 
											int Keytype, 
											LPSTR CAKey, 
											BOOL ShowKeyDetails,
											UINT DisplayType,
                							UINT SharedButtonID,
											LPSTR KeyRingName);
                                
int __declspec(dllexport) FAR PASCAL SimplePGPEncryptFile(
                                          HWND hWnd1,
                                          LPSTR InputFileName,
                                          LPSTR OutputFileName,
                                          BOOL SignIt,
                                          BOOL Wipe,
                                          BOOL Armor,
                                          BOOL TextMode,
                                          BOOL IDEAOnly,
                                          BOOL UseUntrustedKeys,
                                          LPSTR RecipientList,
                                          LPSTR SignerKeyID,
                                          int SignerBufferLen,
                                          LPSTR SignerPassphrase,
                                          int SignerPwdBufferLen,
                                          LPSTR IDEAPassphrase,
                                          int IDEAPwdBufferLen,
                                          LPSTR PublicKeyRingName,
                                          LPSTR PrivateKeyRingName);
                    

int __declspec(dllexport) FAR PASCAL SimplePGPEncryptBuffer(
                                          HWND hWnd1,
				                          LPSTR InputBuffer,
				                          long InputBufferLen,
				                          LPSTR InputBufferName,
				                          LPSTR OutputBuffer,
				                          long FAR *pOutputBufferLen,
				                          BOOL SignIt,
				                          BOOL Armor,
				                          BOOL TextMode,
				                          BOOL IDEAOnly,
				                          BOOL UseUntrustedKeys,
				                          LPSTR RecipientList,
				                          LPSTR SignerKeyID,
				                          int SignerBufferLen,
				                          LPSTR SignerPassphrase,
				                          int SignerPwdBufferLen,
				                          LPSTR IDEAPassphrase,
				                          int IDEAPwdBufferLen,
				                          LPSTR PublicKeyRingName,
				                          LPSTR PrivateKeyRingName);
                    
int __declspec(dllexport) FAR PASCAL SimplePGPSignFile(
											HWND hWnd1,
											LPSTR InputFileName,
											LPSTR OutputFileName,
											BOOL Armor,
											BOOL TextMode,
											BOOL SignatureOnly,
											BOOL ClearSign,
											LPSTR SignerKeyID,
											int SignerBufferLen,
											LPSTR SignerPassphrase,
											int SignerPwdBufferLen,
											LPSTR PrivateKeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPSignBuffer(
											HWND hWnd1,
											LPSTR InputBuffer,
											long InputBufferLen,
				                            LPSTR InputBufferName,
											LPSTR OutputBuffer,
											long FAR *pOutputBufferLen,
											BOOL Armor,
											BOOL TextMode,
											BOOL SignatureOnly,
											BOOL ClearSign,
											LPSTR SignerKeyID,
											int SignerBufferLen,
											LPSTR SignerPassphrase,
											int SignerPwdBufferLen,
											LPSTR PrivateKeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPVerifyDetachedSignatureFile(
											HWND hWnd1,
											LPSTR InputDataFileName,
											LPSTR InputSignatureFileName,
											int FAR *SignatureStatus,
											LPSTR Signer,                        
											int SignerBufLen,    
											LPSTR SignDate,
											int SignDateBufLen,
											LPSTR PublicKeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPVerifyDetachedSignatureBuffer(
											HWND hWnd1,
											LPSTR InputDataBuffer,
											long InputDataBufferLen,
											LPSTR InputSignatureBuffer,
											long InputSignatureBufferLen,
											int FAR *SignatureStatus,
											LPSTR Signer,                        
											int SignerBufLen,    
											LPSTR SignDate,
											int SignDateBufLen,
											LPSTR PublicKeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPReceiveFile(
                							HWND hWnd1,
											LPSTR InputFileName,
											LPSTR OutputFileName,
											LPSTR DecryptPassphrase,
											int DecryptPwdBufferLen,
											int FAR *SignatureStatus,
											LPSTR Signer,                        
											int SignerBufLen,    
											LPSTR SignDate,
											int SignDateBufLen,
											LPSTR PublicKeyRingName,
											LPSTR PrivateKeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPReceiveBuffer(
                							HWND hWnd1,
											LPSTR InputBuffer,
											long InputBufferLen,
				                            LPSTR InputBufferName,
											LPSTR OutputBuffer,
											long FAR *pOutputBufferLen,
											LPSTR DecryptPassphrase,
											int DecryptPwdBufferLen,
											int FAR *SignatureStatus,
											LPSTR Signer,                        
											int SignerBufLen,    
											LPSTR SignDate,
											int SignDateBufLen,
											LPSTR PublicKeyRingName,
											LPSTR PrivateKeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPGenerateKey(
											HWND hWnd,
											LPSTR UserID,
											int KeyLength,
											int KeyType,
											int ValidDays,
											LPSTR Passphrase,
											int PwdBufferLen,
											LPSTR PublicKeyRingName,
											LPSTR PrivateKeyRingName,
											LPSTR GeneratedKeyID);

int __declspec(dllexport) FAR PASCAL SimplePGPExtractKey(
											HWND hWnd1,
											LPSTR UserIDKeyID,
											LPSTR OutputFileName,
											LPSTR KeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPAddKey(
											HWND hWnd1,
											LPSTR InputFileName,
											LPSTR KeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPOpenPublicKeyRing(
											HWND hWnd1,
											LPSTR PublicKeyRIngName);

int __declspec(dllexport) FAR PASCAL SimplePGPGetNextPublicKey(
											HWND hWnd1,
											LPSTR UserID, 
											LPSTR KeyID, 
											int FAR *KeyLength, 
											LPSTR CreationDate, 
											LPSTR ExpirationDate, 
											int FAR *ValidDays,
											int FAR *KeyType,
											LPSTR KeyTypeES,		// "es", "e ", or "s "
											LPSTR KeyState);           // "exp", "rev", "dis", "   " for current
																					// or "days to expire", "  0" to "999"

int __declspec(dllexport) FAR PASCAL SimplePGPClosePublicKeyRing(HWND hWnd1);

int __declspec(dllexport) FAR PASCAL SimplePGPOpenPrivateKeyRing(
											HWND hWnd1,
											LPSTR PrivateKeyRIngName);

int __declspec(dllexport) FAR PASCAL SimplePGPGetNextPrivateKey(
											HWND hWnd1,
											LPSTR UserID, 
											LPSTR KeyID, 
											int FAR *KeyLength, 
											LPSTR CreationDate, 
											LPSTR ExpirationDate, 
											int FAR *ValidDays,
											int FAR *KeyType,
											LPSTR KeyTypeES,		// "es", "e ", or "s "
											LPSTR KeyState);           // "exp", "rev", "dis", "   " for current
																					// or "days to expire", "  0" to "999"

int __declspec(dllexport) FAR PASCAL SimplePGPClosePrivateKeyRing(HWND hWnd1);

int __declspec(dllexport) FAR PASCAL SimplePGPWipeFile(LPSTR FileName);

int __declspec(dllexport) FAR PASCAL SimplePGPRemoveKey(
											HWND hWnd1,
											LPSTR UserIDKeyID,
											LPSTR KeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPCertifyPublicKey(
											HWND hWnd1,
											LPSTR PublicKeyToCertify,
											LPSTR SignerKeyID,
											int SignerBufferLen,
											LPSTR SignerPassphrase,
											int SignerPwdBufferLen,
											LPSTR PublicKeyRingName,
											LPSTR PrivateKeyRingName);

long __declspec(dllexport) FAR PASCAL SimplePGPCountPublicKeys(LPSTR PublicKeyRingName);

long __declspec(dllexport) FAR PASCAL SimplePGPCountPrivateKeys(LPSTR PrivateKeyRingName);

int __declspec(dllexport) FAR PASCAL SimplePGPAnalyzeFile(LPSTR FileName);

int __declspec(dllexport) FAR PASCAL SimplePGPAnalyzeBuffer(LPSTR Buffer, long Len);

int __declspec(dllexport) FAR PASCAL SimplePGPGetVersion(LPSTR Buffer, int Len);

// signature status codes
#define SIGSTS_NOTSIGNED  1
#define SIGSTS_VERIFIED  2
#define SIGSTS_NOTVERIFIED  3
#define SIGSTS_BADSIG  4
#define SIGSTS_VERIFIED_UNTRUSTED  5

/***********************************************************

	Simple PGP Return Codes
	
	A zero return code always means no error.
	Otherwise all Simple PGP return codes 
	have unique values.


************************************************************/
// PGP kernel return codes
#define KERNEL_EXIT_OK				0
//#define KERNEL_EXIT_OK_0			0
					/* for +batchmode, file without a */
					/* valid signature - EXIT_OK_NOSIG */
#define KERNEL_EXIT_OK_NOSIG			1
//#define KERNEL_INVALID_FILE_ERROR		1
#define KERNEL_FILE_NOT_FOUND_ERROR		2
#define KERNEL_UNKNOWN_FILE_ERROR		3
#define KERNEL_NO_BATCH			4
#define KERNEL_BAD_ARG_ERROR			5
#define KERNEL_INTERRUPT			6
#define KERNEL_OUT_OF_MEM			7
#define KERNEL_INVALID_FILE_OP_ERROR		8

/* Keyring errors: Base value = 10 */
#define KERNEL_KEYGEN_ERROR			10
#define KERNEL_NONEXIST_KEY_ERROR		11
#define KERNEL_KEYRING_ADD_ERROR		12
#define KERNEL_KEYRING_EXTRACT_ERROR		13
#define KERNEL_KEYRING_EDIT_ERROR		14
#define KERNEL_KEYRING_VIEW_ERROR		15
#define KERNEL_KEYRING_REMOVE_ERROR		16
#define KERNEL_KEYRING_CHECK_ERROR		17
#define KERNEL_KEY_SIGNATURE_ERROR		18
#define KERNEL_KEYSIG_REMOVE_ERROR		19

/* Encode errors: Base value = 20 */
#define KERNEL_SIGNATURE_ERROR			20
#define KERNEL_RSA_ENCR_ERROR			21
#define KERNEL_ENCR_ERROR			22
#define KERNEL_COMPRESS_ERROR			23

/* Decode errors: Base value = 30 */
#define KERNEL_SIGNATURE_CHECK_ERROR		30
#define KERNEL_RSA_DECR_ERROR			31
#define KERNEL_DECR_ERROR			32
#define KERNEL_DECOMPRESS_ERROR		33

/* PGP-detected problems */
					/* Unexpected conditions found */
					/* (Use this code if no other code */
					/* fits, and the conditions seem so */
					/* wildly unlikely as to not need */
					/* a unique exit code - */
					/* expect the unexpected.) */
#define KERNEL_EXIT_UNEXPECTED		80
					/* Evaluation copy of executable */
					/* expire date has passed. */
#define KERNEL_EXIT_EXPIRED		81

					/* -r pgpoutfile */
					/* syntax or cannot open */
#define KERNEL_EXIT_STDOUT_FAIL	82
					/* @cmdfile or -@ cmdfile */
					/* syntax or cannot open */
#define KERNEL_EXIT_CMDFILE_OPEN	83
					/* @cmdfile mem alloc failure */
					/* arg list too big */
					/* avail mem too small */
					/* or ctrl value problem */
#define KERNEL_EXIT_CMDFILE_MEM	84
					/* @cmdfile unrecognized directive */
#define KERNEL_EXIT_CMDFILE_UNREC	85
					/* OK exit status, non-zero value */
					/* used with setjmp()/longjmp() */
					/* to resume pgp_cmd_ln()/main() */
#define KERNEL_EXIT_OK_NZ		86
					/* Problem with kbhit() */
					/* simulation - raw terminal IO */
#define KERNEL_EXIT_TERM_IO		87
					/* Attempt to create session with */
					/* cryptographic engine - failed */
#define KERNEL_NO_CRYPT_ENGINE		88
#define KERNEL_VERSION_MISMATCH		90
#define KERNEL_CANCEL_ON_BUSY		91

#define KERNEL_EXIT_CODE_NOT_WRITTEN	100

// keysel return codes
#define KEYSEL_OK          0
#define KEYSEL_LOCKED      201
#define KEYSEL_BADKEYRINGTYPE 202  // The KeyRingType argument is invalid
#define KEYSEL_BADKEYTYPE     203
#define KEYSEL_NOKEYRINGNAME  204
#define KEYSEL_USERABORT      205
#define KEYSEL_NOPGPPATH   206     // No PGPPATH env variable set
#define KEYSEL_OPENFAILED  207     // fopen of keyring failed
#define KEYSEL_EOF         208     // End-Of-File in keyring encountered
#define KEYSEL_BADCTBLENOFLEN  209 // Unexpected CTB length-of-length code 
                                  //in ringfile.  Possible keyring corruption
#define KEYSEL_READFAILURE 210     // fread of ringfile failed unexpectedly
#define KEYSEL_BADUIDLEN   211     // User ID longer than 255 (bad!)
#define KEYSEL_UIDTOOLONG  212     // the length of the userid is longer
                                  // than the size of the userid element
                                  // of the pgpkey struct.
#define KEYSEL_BADCREATIONDATE  213 // Creation date in key is 0
#define KEYSEL_UNKNOWNUID 214
#define KEYSEL_NOMEMORY   215   // malloc failed
#define KEYSEL_BADPKT     216   // unexpected for malformed packet found
#define KEYSEL_BADPOINTER 217   // one of the pointer args to KeySel is invalid
#define KEYSEL_USEIDEAONLY 218
#define KEYSEL_CANCEL_ON_BUSY 219

#define KEYSEL_UNKNOWN 225

// Simple PGP function return codes
#define SIMPLEPGPKEYSEL_KEYLIBOK  0
#define SIMPLEPGPKEYSEL_USEIDEAONLY  301
#define SIMPLEPGPKEYSEL_CANCEL  302
#define SIMPLEPGPKEYSEL_NULLCAPTIONSTRINGPOINTER  303
#define SIMPLEPGPKEYSEL_EMPTYCAPTIONSTRING  304
#define SIMPLEPGPKEYSEL_KEYRINGTYPENOTPUBLICORSECRET  305
#define SIMPLEPGPKEYSEL_NULLPOINTERTOSELECTEDLIST  306
#define SIMPLEPGPKEYSEL_SELECTEDLISTLENGTHTOOSMALL  307
#define SIMPLEPGPKEYSEL_NULLPOINTERTOSELECTEDLISTCOUNT  308
#define SIMPLEPGPKEYSEL_BADINCLUDEWHATVALUE  309
#define SIMPLEPGPKEYSEL_NULLPOINTERTOUIDKIDSEARCHSTRING  310
#define SIMPLEPGPKEYSEL_BADKEYTYPEBITMAP  311
#define SIMPLEPGPKEYSEL_BADDISPLAYTYPEBITMAP  312
#define SIMPLEPGPKEYSEL_BADSHAREDBUTTONIDBITMAP  313
#define SIMPLEPGPKEYSEL_KEYRINGFILENOTFOUND  314

#define SIMPLEPGPENCRYPTFILE_OK  0
#define SIMPLEPGPENCRYPTFILE_NULLPOINTERTOINPUTFILENAME   401
#define SIMPLEPGPENCRYPTFILE_EMPTYINPUTFILENAMESTRING   402
#define SIMPLEPGPENCRYPTFILE_INPUTFILEDOESNOTEXIST   403
#define SIMPLEPGPENCRYPTFILE_NULLPOINTERTOOUTPUTFILENAME   404
#define SIMPLEPGPENCRYPTFILE_EMPTYOUTPUTFILENAMESTRING   405
#define SIMPLEPGPENCRYPTFILE_NULLPOINTERTORECIPIENTLIST   406
#define SIMPLEPGPENCRYPTFILE_EMPTYRECIPIENTLISTSTRING   407
#define SIMPLEPGPENCRYPTFILE_RECIPIENTLISTDOESNOTENDWITHNEWLINE    408
#define SIMPLEPGPENCRYPTFILE_RECIPIENTLISTDOESNOTSTARTWITHGOODCODECHAR   409
#define SIMPLEPGPENCRYPTFILE_NOTENOUGHMEMORYFORINPUTSTRUCTURE    413
#define SIMPLEPGPENCRYPTFILE_NOTENOUGHMEMORYFOROUTPUTSTRUCTURE   414
#define SIMPLEPGPENCRYPTFILE_KEYSELCANCEL    415
#define SIMPLEPGPENCRYPTFILE_SIGNERPWDBUFFERTOOSMALL    416
#define SIMPLEPGPENCRYPTFILE_IDEAPWDBUFFERTOOSMALL    417
#define SIMPLEPGPENCRYPTFILE_CANNOTUSEUNTRUSTEDKEY    418
#define SIMPLEPGPENCRYPTFILE_ENCRYPTNOTSIGN_FUNCTIONNOTENABLED    419
#define SIMPLEPGPENCRYPTFILE_ENCRYPTANDSIGN_FUNCTIONNOTENABLED    420
#define SIMPLEPGPENCRYPTFILE_CANNOTUSEUNCERTIFIEDKEY    421

#define SIMPLEPGPREADRECIPIENTLIST_OK    0
#define SIMPLEPGPREADRECIPIENTLIST_NULLPOINTEROREMPTYLIST    501
#define SIMPLEPGPREADRECIPIENTLIST_LISTDOESNOTSTARTWITHGOODCODECHAR    502
#define SIMPLEPGPREADRECIPIENTLIST_AUSERIDSTRINGISTOOLONG    503
#define SIMPLEPGPREADRECIPIENTLIST_OUTOFMEMORYFORUSERIDLIST    504
#define SIMPLEPGPREADRECIPIENTLIST_AKEYIDSTRINGISTOOLONG    505
#define SIMPLEPGPREADRECIPIENTLIST_OUTOFMEMORYFORKEYIDLIST    506

#define SIMPLEPGPENCRYPTBUFFER_NULLPOINTERTOINPUTBUFFER 601
#define SIMPLEPGPENCRYPTBUFFER_INPUTBUFFERLENGTHISZERO  602
#define SIMPLEPGPENCRYPTBUFFER_NULLPOINTERTOOUTPUTBUFFER  603
#define SIMPLEPGPENCRYPTBUFFER_NULLPOINTERTORECIPIENTLIST  604
#define SIMPLEPGPENCRYPTBUFFER_EMPTYRECIPIENTLISTSTRING  605
#define SIMPLEPGPENCRYPTBUFFER_RECIPIENTLISTDOESNOTENDWITHNEWLINE  606
#define SIMPLEPGPENCRYPTBUFFER_RECIPIENTLISTDOESNOTSTARTWITHGOODCODECHAR  607
#define SIMPLEPGPENCRYPTBUFFER_NOTENOUGHMEMORYFORINPUTSTRUCTURE  611
#define SIMPLEPGPENCRYPTBUFFER_NOTENOUGHMEMORYFOROUTPUTSTRUCTURE  612
#define SIMPLEPGPENCRYPTBUFFER_KEYSELCANCEL  613
#define SIMPLEPGPENCRYPTBUFFER_SIGNERPWDBUFFERTOOSMALL  614
#define SIMPLEPGPENCRYPTBUFFER_IDEAPWDBUFFERTOOSMALL  615
#define SIMPLEPGPENCRYPTBUFFER_OUTPUTBUFFERTOOSMALL  616
#define SIMPLEPGPENCRYPTBUFFER_CANNOTUSEUNTRUSTEDKEY  617
#define SIMPLEPGPENCRYPTBUFFER_ENCRYPTNOTSIGN_FUNCTIONNOTENABLED  618
#define SIMPLEPGPENCRYPTBUFFER_ENCRYPTANDSIGN_FUNCTIONNOTENABLED  619
#define SIMPLEPGPENCRYPTBUFFER_CANNOTUSEUNCERTIFIEDKEY    620
#define SIMPLEPGPENCRYPTBUFFER_16BITINPUTBUFFERLENGTHGTR65500    621
#define SIMPLEPGPENCRYPTBUFFER_16BITOUTPUTBUFFERLENGTHGTR65500    622

#define SIMPLEPGPSIGNFILE_NULLPOINTERTOINPUTFILENAME  701
#define SIMPLEPGPSIGNFILE_EMPTYINPUTFILENAMESTRING  702
#define SIMPLEPGPSIGNFILE_INPUTFILEDOESNOTEXIST  703
#define SIMPLEPGPSIGNFILE_NULLPOINTERTOOUTPUTFILENAME  704
#define SIMPLEPGPSIGNFILE_EMPTYOUTPUTFILENAMESTRING  705
#define SIMPLEPGPSIGNFILE_NOTENOUGHMEMORYFORINPUTSTRUCTURE  708
#define SIMPLEPGPSIGNFILE_NOTENOUGHMEMORYFOROUTPUTSTRUCTURE  709
#define SIMPLEPGPSIGNFILE_KEYSELCANCEL  710
#define SIMPLEPGPSIGNFILE_SIGNERPWDBUFFERTOOSMALL  711
#define SIMPLEPGPSIGNFILE_FUNCTIONNOTENABLED  712
#define SIMPLEPGPSIGNFILE_CANNOTUSEUNCERTIFIEDKEY    713

#define SIMPLEPGPSIGNBUFFER_NULLPOINTERTOINPUTBUFFER  801
#define SIMPLEPGPSIGNBUFFER_INPUTBUFFERLENGTHISZERO  802
#define SIMPLEPGPSIGNBUFFER_NULLPOINTERTOOUTPUTBUFFER  803
#define SIMPLEPGPSIGNBUFFER_NOTENOUGHMEMORYFORINPUTSTRUCTURE  806
#define SIMPLEPGPSIGNBUFFER_NOTENOUGHMEMORYFOROUTPUTSTRUCTURE  807
#define SIMPLEPGPSIGNBUFFER_KEYSELCANCEL  808
#define SIMPLEPGPSIGNBUFFER_SIGNERPWDBUFFERTOOSMALL  809
#define SIMPLEPGPSIGNBUFFER_OUTPUTBUFFERTOOSMALL  810
#define SIMPLEPGPSIGNBUFFER_FUNCTIONNOTENABLED  811
#define SIMPLEPGPSIGNBUFFER_CANNOTUSEUNCERTIFIEDKEY    812
#define SIMPLEPGPSIGNBUFFER_16BITINPUTBUFFERLENGTHGTR65500    813
#define SIMPLEPGPSIGNBUFFER_16BITOUTPUTBUFFERLENGTHGTR65500    814

#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_NULLPOINTERTOINPUTDATAFILENAME  901
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_EMPTYINPUTDATAFILENAMESTRING  902
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_INPUTDATAFILEDOESNOTEXIST  903
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_NULLPOINTERTOINPUTSIGFILENAME  904
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_EMPTYINPUTSIGFILENAMESTRING  905
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_INPUTSIGFILEDOESNOTEXIST  906
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_NULLPOINTERTOSIGNATURESTATUS  907
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_NULLPOINTERTOSIGNERSTRING  908
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_SIGNERBUFLENGTHTOOSMALL  909
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_NULLPOINTERTOSIGNDATE  910
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_SIGNDATEBUFLENGTHTOOSMALL  911
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_NOTENOUGHMEMORYFORINPUTDATASTRUCTURE  912
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_NOTENOUGHMEMORYFORINPUTSIGSTRUCTURE  913
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_FUNCTIONNOTENABLED  914
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREFILE_CANNOTUSEUNCERTIFIEDKEY    915

#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_NULLPOINTERTOINPUTDATABUFFER  1001
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_INPUTDATABUFFERLENGTHISZERO  1002
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_NULLPOINTERTOINPUTSIGBUFFER  1003
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_INPUTSIGBUFFERLENGTHISZERO  1004
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_NULLPOINTERTOSIGNATURESTATUS  1005
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_NULLPOINTERTOSIGNERSTRING  1006
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_SIGNERBUFLENGTHTOOSMALL  1007
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_NULLPOINTERTOSIGNDATE  1008
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_SIGNDATEBUFLENGTHTOOSMALL  1009
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_NOTENOUGHMEMORYFORINPUTDATASTRUCTURE  1010
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_NOTENOUGHMEMORYFORINPUTSIGSTRUCTURE  1011
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_FUNCTIONNOTENABLED  1012
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_CANNOTUSEUNCERTIFIEDKEY    1013
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_16BITINPUTDATABUFFERLENGTHGTR65500    1014
#define SIMPLEPGPVERIFYDETACHEDSIGNATUREBUFFER_16BITINPUTSIGBUFFERLENGTHGTR65500    1015

#define SIMPLEPGPRECEIVEFILE_NULLPOINTERTOINPUTFILENAME  1101
#define SIMPLEPGPRECEIVEFILE_EMPTYINPUTFILENAMESTRING  1102
#define SIMPLEPGPRECEIVEFILE_INPUTFILEDOESNOTEXIST  1103
#define SIMPLEPGPRECEIVEFILE_NULLPOINTERTOOUTPUTFILENAME  1104
#define SIMPLEPGPRECEIVEFILE_EMPTYOUTPUTFILENAMESTRING  1105
#define SIMPLEPGPRECEIVEFILE_NULLPOINTERTOSIGNATURESTATUS  1107
#define SIMPLEPGPRECEIVEFILE_NULLPOINTERTOSIGNERSTRING  1108
#define SIMPLEPGPRECEIVEFILE_SIGNERBUFLENGTHTOOSMALL  1109
#define SIMPLEPGPRECEIVEFILE_NULLPOINTERTOSIGNDATE  1110
#define SIMPLEPGPRECEIVEFILE_SIGNDATEBUFLENGTHTOOSMALL  1111
#define SIMPLEPGPRECEIVEFILE_NOTENOUGHMEMORYFORINPUTSTRUCTURE  1112
#define SIMPLEPGPRECEIVEFILE_NOTENOUGHMEMORYFOROUTPUTSTRUCTURE  1113
#define SIMPLEPGPRECEIVEFILE_DECRYPTPWDBUFFERTOOSMALL  1114
#define SIMPLEPGPRECEIVEFILE_ISKEY  1115
#define SIMPLEPGPRECEIVEFILE_FUNCTIONNOTENABLED  1116
#define SIMPLEPGPRECEIVEFILE_CANNOTUSEUNCERTIFIEDKEY    1117
#define SIMPLEPGPRECEIVEFILE_OUTPUTFILENAMETOOLONG    1118

#define SIMPLEPGPRECEIVEBUFFER_NULLPOINTERTOINPUTBUFFER  1201
#define SIMPLEPGPRECEIVEBUFFER_INPUTBUFFERLENGTHISZERO  1202
#define SIMPLEPGPRECEIVEBUFFER_NULLPOINTERTOOUTPUTBUFFER  1203
#define SIMPLEPGPRECEIVEBUFFER_NULLPOINTERTOSIGNATURESTATUS  1205
#define SIMPLEPGPRECEIVEBUFFER_NULLPOINTERTOSIGNERSTRING  1206
#define SIMPLEPGPRECEIVEBUFFER_SIGNERBUFLENGTHTOOSMALL  1207
#define SIMPLEPGPRECEIVEBUFFER_NULLPOINTERTOSIGNDATE  1208
#define SIMPLEPGPRECEIVEBUFFER_SIGNDATEBUFLENGTHTOOSMALL  1209
#define SIMPLEPGPRECEIVEBUFFER_NOTENOUGHMEMORYFORINPUTSTRUCTURE  1210
#define SIMPLEPGPRECEIVEBUFFER_NOTENOUGHMEMORYFOROUTPUTSTRUCTURE  1211
#define SIMPLEPGPRECEIVEBUFFER_DECRYPTPWDBUFFERTOOSMALL  1212
#define SIMPLEPGPRECEIVEBUFFER_OUTPUTBUFFERTOOSMALL  1213
#define SIMPLEPGPRECEIVEBUFFER_ISKEY  1214
#define SIMPLEPGPRECEIVEBUFFER_FUNCTIONNOTENABLED  1215
#define SIMPLEPGPRECEIVEBUFFER_CANNOTUSEUNCERTIFIEDKEY    1216
#define SIMPLEPGPRECEIVEBUFFER_16BITINPUTBUFFERLENGTHGTR65500    1217
#define SIMPLEPGPRECEIVEBUFFER_16BITOUTPUTBUFFERLENGTHGTR65500    1218

#define SIMPLEPGPGENERATEKEY_FUNCTIONNOTENABLED  1301
#define SIMPLEPGPGENERATEKEY_NULLPOINTERTOUSEIDSTRING  1302
#define SIMPLEPGPGENERATEKEY_USERIDSTRINGISEMPTY  1303
#define SIMPLEPGPGENERATEKEY_KEYLENGTHOUTOFRANGE  1304
#define SIMPLEPGPGENERATEKEY_BADKEYTYPE  1305
#define SIMPLEPGPGENERATEKEY_VALIDDAYSOUTOFRANGE  1306
#define SIMPLEPGPGENERATEKEY_NULLPOINTERTOGENKEYID  1308
#define SIMPLEPGPGENERATEKEY_PWDBUFFERTOOSMALL  1309

#define SIMPLEPGPEXTRACTKEY_FUNCTIONNOTENABLED  1401
#define SIMPLEPGPEXTRACTKEY_NULLPOINTERTOUIDKIDSTRING  1402
#define SIMPLEPGPEXTRACTKEY_UIDKIDSTRINGISEMPTY  1403
#define SIMPLEPGPEXTRACTKEY_NULLPOINTERTOOUTPUTFILENAME  1404
#define SIMPLEPGPEXTRACTKEY_OUTPUTFILENAMESTRINGISEMPTY  1405
#define SIMPLEPGPEXTRACTKEY_NOTENOUGHMEMORYFOROUTPUTSTRUCTURE  1406

#define SIMPLEPGPADDKEY_FUNCTIONNOTENABLED  1501
#define SIMPLEPGPADDKEY_NULLPOINTERTOINPUTFILENAME  1502
#define SIMPLEPGPADDKEY_INPUTFILENAMESTRINGISEMPTY  1503
#define SIMPLEPGPADDKEY_INPUTFILENAMEDOESNOTEXIST  1504
#define SIMPLEPGPADDKEY_NOTENOUGHMEMORYFORINPUTSTRUCTURE  1505

#define SIMPLEPGPREMOVEKEY_FUNCTIONNOTENABLED  1601
#define SIMPLEPGPREMOVEKEY_NULLPOINTERTOUIDKIDSTRING  1602
#define SIMPLEPGPREMOVEKEY_UIDKIDSTRINGISEMPTY  1603

#define SIMPLEPGPCERTIFYKEY_FUNCTIONNOTENABLED  1701
#define SIMPLEPGPCERTIFYKEY_NULLPOINTERTOKEYTOCERTIFY  1702
#define SIMPLEPGPCERTIFYKEY_KEYTOCERTIFYSTRINGISEMPTY  1703
#define SIMPLEPGPCERTIFYKEY_NULLPOINTERTOSIGNERSTRING  1704
#define SIMPLEPGPCERTIFYKEY_KEYSELCANCEL  1706
#define SIMPLEPGPCERTIFYKEY_SIGNERPWDBUFFERTOOSMALL  1707
#define SIMPLEPGPCERTIFYKEY_CANNOTUSEUNCERTIFIEDKEY    1708

#define SIMPLEPGPOPENPUBLICKEYRING_OK  0
#define SIMPLEPGPOPENPUBLICKEYRING_USERCANCELONTASKBUSY  1801
#define SIMPLEPGPOPENPUBLICKEYRING_CANTOPENKEYRINGFILE  1802

#define SIMPLEPGPGETNEXTPUBLICKEY_OK  0
#define SIMPLEPGPGETNEXTPUBLICKEY_EOF  8
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLUSERIDSTRINGPOINTER  1901
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLKEYIDSTRINGPOINTER  1902
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLKEYLENGTHPOINTER  1903
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLCREATIONDATEPOINTER  1904
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLEXPIRATIONDATEPOINTER  1905
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLVALIDDAYSPOINTER  1906
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLKEYTYPEPOINTER  1907
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLKEYTYPEESPOINTER  1908
#define SIMPLEPGPGETNEXTPUBLICKEY_NULLKEYSTATEPOINTER  1909

#define SIMPLEPGPCLOSEPUBLICKEYRING_OK  0

#define SIMPLEPGPOPENPRIVATEKEYRING_OK  0
#define SIMPLEPGPOPENPRIVATEKEYRING_USERCANCELONTASKBUSY  2001
#define SIMPLEPGPOPENPRIVATEKEYRING_CANTOPENKEYRINGFILE  2002

#define SIMPLEPGPGETNEXTPRIVATEKEY_OK  0
#define SIMPLEPGPGETNEXTPRIVATEKEY_EOF  8
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLUSERIDSTRINGPOINTER  2101
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLKEYIDSTRINGPOINTER  2102
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLKEYLENGTHPOINTER  2103
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLCREATIONDATEPOINTER  2104
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLEXPIRATIONDATEPOINTER  2105
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLVALIDDAYSPOINTER  2106
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLKEYTYPEPOINTER  2107
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLKEYTYPEESPOINTER  2108
#define SIMPLEPGPGETNEXTPRIVATEKEY_NULLKEYSTATEPOINTER  2109

#define SIMPLEPGPCLOSEPRIVATEKEYRING_OK  0

// analyze return codes
#define SIMPLEANALYZE_UNKNOWN  0
#define SIMPLEANALYZE_ENCR  1
#define SIMPLEANALYZE_SIGN  2
#define SIMPLEANALYZE_KEY  3
#define SIMPLEANALYZE_IOERROR  4

#define SIMPLEPGPANALYZEFILE_FILENAMEPOINTERISNULL 2201
#define SIMPLEPGPANALYZEFILE_FILENAMESTRINGISEMPTY 2202
#define SIMPLEPGPANALYZEFILE_FILENAMEDOESNOTEXIST 2203

#define SIMPLEPGPANALYZEBUFFER_BUFFERPOINTERISNULL  2301
#define SIMPLEPGPANALYZEBUFFER_BUFFERLENGTHISZERO     2302
#define SIMPLEPGPANALYZEBUFFER_CANTOPENTEMPFILE    2304
#define SIMPLEPGPANALYZEBUFFER_ERRORWRITINGTEMPFILE    2305

#define SIMPLEPGPWIPEFILE_OK  0
#define SIMPLEPGPWIPEFILE_CANTOPENFILE  2401
#define SIMPLEPGPWIPEFILE_CANTREMOVEFILE  2402

#define SIMPLEPGPGETVERSION_OK  0
#define SIMPLEPGPGETVERSION_BUFFERPOINTERISNULL   2501
#define SIMPLEPGPGETVERSION_BUFFERTOOSMALL  2502

#define SIMPLEPGP_OK 0
#define SIMPLEPGP_OUTOFMEM 2600
#define SIMPLEPGP_UNABLETOMATCHID 2601

/*******************************************************************
The following functions are for getting blocks; the Simple API is not 
designed to handle more than one PGP block at a time.
*******************************************************************/

typedef unsigned long PGPBLOCKERR;

#define PGP_BLOCK_SUCCESS			0
#define PGP_BLOCK_ERR_BAD_PARAM     1

/*See "Putting it all together" at the bottom for the way it integrates into
 *existing code:
 */

/*These functions return information (and pointers to) the BEGIN PGP blocks
 *included in buffers passed in.
 */

//First of all, the types of blocks that are supported:

#define PGP_BLOCK_TYPE_UNKNOWN      0
#define PGP_BLOCK_TYPE_ENCRYPTED    1
//(Encrypted and encrypted and signed are considered the same).
#define PGP_BLOCK_TYPE_CLEARSIGNED  2
#define PGP_BLOCK_TYPE_KEY          4
#define PGP_BLOCK_TYPE_NON_PGP      8
//Non-PGP is for non-PGP text that is between PGP blocks.
#define PGP_BLOCK_TYPE_ANY_PGP      (PGP_BLOCK_TYPE_ENCRYPTED   |\
									 PGP_BLOCK_TYPE_CLEARSIGNED |\
									 PGP_BLOCK_TYPE_KEY)

#define PGP_BLOCK_TYPE_ANY          (PGP_BLOCK_TYPE_ANY_PGP |\
									 PGP_BLOCK_TYPE_NON_PGP)

/***********************************************************************
This function looks through the data in Buffer.  It returns the first
block found whose block type matches the BlockMask.  The actual type of
the block is placed in BlockType, and the length in BlockSize.  The
pointer NextBlock is set to point to the offset in Buffer where the
the block begins.  If there is no corresponding block to the block
mask (ie, there isn't a "----PGP BEGIN" of the requested type)
then NextBlock is set to NULL.
See "Putting it All Together" for an example.

There is one special case to be discussed - if there is a known
"----PGP BEGIN..." header, with no corresponding "----END" footer,
NextBlock still contains a pointer to that block, and BlockSize is set
to be the remaining length of the file.
 ***********************************************************************/
PGPBLOCKERR __declspec(dllexport) FAR PASCAL 
	SimplePGPGetFirstBlock(char **NextBlock,
						   unsigned long *BlockSize,
						   unsigned short *BlockType,
						   const unsigned short BlockMask,
						   const char *Buffer);

/************************************************************************
This functions similarly to PGPGetFirstBlock. The basic difference is
that it gets the NEXT block (never would've guessed from the name,
eh?)  The idea being that you call GetFirst() and then loop on
GetNext().  See PGPGetFirstBlock for details on the parameters.  Note
that NextBlock should be pointint to the current location in the
buffer.  See "Putting It All Together," below.
 ***********************************************************************/
PGPBLOCKERR __declspec(dllexport) FAR PASCAL 
	SimplePGPGetNextBlock(char **NextBlock,
						  unsigned long *BlockSize,
						  unsigned short *BlockType,
						  const unsigned short BlockMask);

/************************************************************************
 This is the underlying function.  Its arguments are basically identical
 to PGPGetFirstBlock, except for the last one - IsGetFirst is either
 1 or 0 to specify whether this function is to be used as a GetFirst
 or a GetNext.  It's not generally reccomended this be used in your
 application; the interface for First/Next is much clearer.
 ***********************************************************************/
PGPBLOCKERR __declspec(dllexport) FAR PASCAL 
	SimplePGPBlockGetFirstOrNextBlock(char **NextBlock, 
									  unsigned long *BlockSize,
									  unsigned short *BlockType,
									  const unsigned short BlockMask,
									  const char *Buffer,
									  int IsGetFirst);

/************************************************************************
                          Putting It All Together

  Here's a sample of what a function to call this stuff would look like:

int decrypt(char *InputBuffer, char *OutputBuffer)
{
	char *Next;
	unsigned long Size;
	unsigned short Type;
	int NumberOfBlocks = 0;

    //Get the first block, if any...
	if((PGPGetFirstBlock(&Next, 
						 &Size, 
						 &Type, 
						 PGP_BLOCK_TYPE_ANY, 
						 InputBuffer) == PGP_BLOCK_SUCCESS) 
						 && Next)
	{
		//Ok, we've got our first block.
		do
		{
			//Process any keys we get...
			if((Type & PGP_BLOCK_TYPE_KEY) == PGP_BLOCK_TYPE_KEY)
			{
				ProcessKeyBlock(Next, Size)
			}
			else  //Otherwise, it's some sort of encrypted or signed block
			{
			    //Obviously, I'm hand-waving a bit on this call:
				SimplePGPReceiveBuffer(...Next, Size, OutputBuffer...);
			}
		}
		while((PGPGetNextBlock(&Next, &Size, &Type, Mask) ==
				PGP_BLOCK_SUCCESS) && Next);
	}
}

  An obvious variant on the above would be to ask for a type of:
  PGP_BLOCK_TYPE_ENCRYPTED | PGP_BLOCK_TYPE_CLEARSIGNED and not do key
  processing in this function.
 ***********************************************************************/

#define SIMPLE_PGP_MAX_ERROR_LEN  512

BOOL __declspec(dllexport) FAR PASCAL 
SimplePGPErrorToString(char *ErrorMessage, 
					   unsigned long ErrorMessageLen,
					   int SimpleResult);


BOOL __declspec(dllexport) FAR PASCAL 
SimplePGPTranslateSignatureResult(char *SignatureResult, 
								  char *Signer, 
								  char *SignatureDate,
								  int SignatureCode);

int __declspec(dllexport) FAR PASCAL SimplePGPSimpleEncrypt(HWND hwnd,
															char **OutputBuffer, 
															char *InputBuffer, 
															char *SignerID,
															BOOL Sign,
															char *RecipientList);


int __declspec(dllexport) FAR PASCAL SimplePGPSimpleSign(HWND hwnd,
														 char **OutputBuffer, 
														 char *InputBuffer,
														 char *SignerID);

int __declspec(dllexport) FAR PASCAL SimplePGPSimpleDecrypt(HWND hwnd,
															char **OutputBuffer,
															char **SignatureResultBuffer,
															char *InputBuffer,
															unsigned long *NumDecryptedBlocks);

void __declspec(dllexport) WINAPI SimplePGPFree(void *p);

#define USERID_LEN  256
#define KEYID_LEN    21
//This is probably way too big, but I don't know what it should be:
#define DATE_LEN    256

typedef struct PublicKey
{
	char UserID[USERID_LEN];
	char KeyID[KEYID_LEN];
	int  KeyLength; //Length of the key, in bits (1024, 2048, etc)
	char CreationDate[DATE_LEN];
	char ExpirationDate[DATE_LEN];
	int  ValidDays; //I have no clue what this is but suspect it's the number of days 'til
	                //the key expires
	int  KeyType;   //Also clueless.
	char KeyTypeES[3];  //"e" for encrypt-only keys, "s" for sign-only keys, and "es" for
						//"standard" encryption and signature keys.
	char KeyState[4];	//"exp" for expired, "rev" for revoked, "dis" for something
						//unspecified, "   " for current and active, or a number
						//from "  0" to "999" for the number of days until the key expires
						//(which might call into doubt my speculation on ValidDays, above).
}PUBLICKEY, *PPUBLICKEY;

