/* server_publickey.c
 *
 * Publickey authentication method
 */

/* lsh, an implementation of the ssh protocol
 *
 * Copyright (C) 1999 Balazs Scheidler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "server_publickey.h"
#include "xalloc.h"
#include "connection.h"
#include "parse.h"
#include "werror.h"
#include "format.h"
#include "charset.h"
#include "ssh.h"
#include "lookup_verifier.h"
#include "server_password.h"

#include "server_publickey.c.x"

/* GABA:
   (class
     (name userauth_publickey)
     (super userauth)
     (vars
       (verifiers object alist)))
*/

static struct lsh_string *format_userauth_pk_ok(int algorithm, struct lsh_string *keyblob)
{
  return ssh_format("%c%a%S", SSH_MSG_USERAUTH_PK_OK, 
		    algorithm, keyblob);
}

static int do_authenticate(struct userauth *i UNUSED,
			   struct ssh_connection *connection,
			   struct lsh_string *username,
			   struct simple_buffer *args,
			   struct command_continuation *c)
{
  CAST(userauth_publickey, self, i);
  struct lsh_string *keyblob = NULL;
  struct lookup_verifier *lookup;
  struct verifier *v;
  UINT8 *signature_blob;
  UINT32 signature_length;
  UINT32 signature_start;
  UINT32 algorithm;
  int check_key;

  username = utf8_to_local(username, 1, 1);
  if (!username)
    return LSH_FAIL | LSH_DIE;

  if (parse_boolean(args, &check_key) &&
      parse_atom(args, &algorithm) &&
      (keyblob = parse_string_copy(args)) &&
      ((check_key && 
	(signature_start = args->pos) && 
	(parse_string(args, &signature_length, &signature_blob)) &&
	parse_eod(args)) || 
       (!check_key && parse_eod(args)))) 
    {
#if DATAFELLOWS_WORKAROUNDS
      if (algorithm == ATOM_SSH_DSS && (connection->peer_flags & PEER_SSH_DSS_KLUDGE))
	{
	  lookup = ALIST_GET(self->verifiers, ATOM_SSH_DSS_KLUDGE);
	}
      else
#endif
	lookup = ALIST_GET(self->verifiers, algorithm);

      if (!lookup) 
	{
	  lsh_string_free(keyblob); 
	  lsh_string_free(username);
	  verbose("Unknown publickey algorithm\n");
	  return COMMAND_RETURN(c, NULL);
	}
      v = LOOKUP_VERIFIER(lookup, username, keyblob);

      if (!check_key)
	{
	  lsh_string_free(username);
	  if (v)
	    {
	      struct lsh_string *reply = format_userauth_pk_ok(algorithm, keyblob);
	      lsh_string_free(keyblob);
	      KILL(v);
	      return A_WRITE(connection->write, reply);
	    }
	  else
	    {
	      lsh_string_free(keyblob);
	      return COMMAND_RETURN(c, NULL);
	    }
	}
      else 
	{
	  struct lsh_string *signed_data;
	  struct unix_user *user = NULL;
#if DATAFELLOWS_WORKAROUNDS
	  /* FIXME: this check should be made more general */
	  if (connection->peer_flags & PEER_USERAUTH_REQUEST_KLUDGE)
	    {
	      signed_data = ssh_format("%lS%c%S%a%a%c%a%S", 
				       connection->session_id,
				       SSH_MSG_USERAUTH_REQUEST,
				       username, 
				       ATOM_SSH_USERAUTH,
				       ATOM_PUBLICKEY,
				       1,
				       algorithm,
				       keyblob);
	    }
	  else
#endif
	      signed_data = ssh_format("%lS%ls", connection->session_id, 
				       signature_start, args->data);

	  lsh_string_free(keyblob); 
	  if (VERIFY(v, signed_data->length, signed_data->data, signature_length, signature_blob))
	    {
	      werror("user %S has authenticated\n", username);
	      user = lookup_user(username, 0);
	    }
	  lsh_string_free(username);
	  lsh_string_free(signed_data);
	  return COMMAND_RETURN(c, user);
	}
    }
  else 
    {
      werror("Badly formatted publickey authentication request\n");
      return LSH_FAIL | LSH_DIE;
    }
}

struct userauth *make_userauth_publickey(struct alist *verifiers)
{
  NEW(userauth_publickey, self);

  self->super.authenticate = do_authenticate;
  self->verifiers = verifiers;
  return &self->super;
}
