//	PGPJN.H
//	Runtime Loadable Encryptor Extension for Pegasus Mail for Windows.
//	Copyright (c) 1995, David Harris, All Rights Reserved.
//
//	Pretty Good Privacy modifications by John Navas
//
//	Copyright (c) 1995, John Navas, All Rights Reserved.
//
//	The author grants explicit permission for this source code to be
//	used or modified as required, subject only to the conditions that
//	the copyright notices above are preserved; that this source code
//	not be used in any product distributed in competition with this
//  product; that by using this code you agree that the code is
//	provided without warranty of any kind, either explicit or implied,
//	and you use it at your own risk.
//
//	This module implements an interface to Pretty Good Privacy (tm)
//	and (c) Copyright 1990-1994 Philip Zimmermann. All rights reserved.
//
//	The Massachusetts Institute of Technology is the distributor of PGP
//	version 2.6, for distribution in the USA only.  It is available from
//	"net-dist.mit.edu," a controlled FTP site that has restrictions and
//	limitations, similar to those used by RSA Data Security, Inc., to comply
//	with export control requirements.  The software resides in the directory
//	/pub/PGP.

#define STRICT
#include <windows.h>
#include <shellapi.h>
#include <ctl3d.h>

#include <ctype.h>
#include <fcntl.h>
#include <io.h>
#include <malloc.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\types.h>
#include <sys\stat.h>

#include "..\wpmforms.h"

///////////////////////////////////////////////////////////////////////////////

#define MINRES	(12)					// minimum system resources needed
#define MAXTAG	(80)					// maximum INI file tag length
#define MAXNAME	(90)					// maximum name length

extern HINSTANCE hLibInstance;			// set in LibMain, used throughout the DLL
extern HWND hwndParent;					// handle of WinPMail parent window
typedef enum { PUB, SIG } MODE;			// my Form mode type
extern MODE MyMode;						// my Form mode
extern char MyDir[_MAX_DRIVE+_MAX_DIR];	// my directory
extern char PGPpath[_MAX_PATH];			// PGP executable
extern char* TEMP;						// TEMP directory for PGPJNDOS

extern char PGPEXE[_MAX_FNAME+_MAX_EXT];	// PGP executable
extern char PGPJNDOS[_MAX_FNAME+_MAX_EXT];	// my special EXE

extern char PGPBegin[MAXTAG];			// start of PGP encrypted section
extern char PGPEnd[MAXTAG];				// end of PGP encrypted section
extern char PGPBeginKey[MAXTAG];		// start of PGP public key
extern char PGPEndKey[MAXTAG];			// end of PGP public key

extern char SigGood[MAXTAG];			// PGP good sig message
extern char SigBad[MAXTAG];				// PGP bad sig message
extern char SigNoKey[MAXTAG];			// PGP missing key to verify sig

extern char PGPBeginAdvise[MAXTAG];		// start of PGP public key
extern char PGPEndAdvise[MAXTAG];		// end of PGP public key

extern char PGPJNKey[128];				// Master Key option

///////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
void _cdecl MyDebugOutput(LPCSTR lpszFmt, ...);
#endif // _DEBUG

///////////////////////////////////////////////////////////////////////////////

char* AddFileName(char* path, const char* file);	// add file name to dir path
char* FrameIt(char* string);						// frame string with blank

///////////////////////////////////////////////////////////////////////////////
// MY CLASS FOR TEMPORARY BUFFER
// automatically deleted when it goes out of scope

class MyBuffer {
	char* buf;
	size_t siz;
public:
	// constructor
	MyBuffer(int req, const char* string = NULL) {
		int len = string ? (lstrlen(string) + 1) : 0 ;
		
		siz = __max(req, len);
		buf = (char*) malloc(siz);
		if (buf != NULL) {
			if (string != NULL)
				lstrcpy(buf, string);
		}
		else {
#ifdef _DEBUG
			MyDebugOutput("malloc failed on %d bytes\n", siz);
#endif // _DEBUG
			siz = 0;
		}
	}
	// destructor
	~MyBuffer() {
		if (buf)
			free(buf);
	}
	// access
	operator char*() {
		return buf;
	}
	size_t GetSize() {
		return siz;
	}
};

///////////////////////////////////////////////////////////////////////////////
// MY CLASS FOR A TEMPORARY FILE
// automatically deleted when it goes out of scope

class MyTempFile {
	char name[_MAX_PATH];
public:
	// constructor
	MyTempFile(char* dir = NULL) {
		OFSTRUCT ofs;
		
		lstrcpy(name, dir ? dir : TEMP);
		AddFileName(name, "JNXXXXXX");
		_mktemp(name);
		HFILE tmp = OpenFile(name, &ofs, OF_WRITE | OF_CREATE);
		if (tmp != HFILE_ERROR)
			_lclose (tmp);
		else
			MessageBox(
				hwndParent,						/* handle of parent window	*/
				_strerror(name),				/* address of text in message box	*/
				"PGP Interface",				/* address of title of message box	*/
				MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
	}
	// destructor
	~MyTempFile() {
		remove(name);
	}
	// access
	operator char*() {
		return name;
	}
};

///////////////////////////////////////////////////////////////////////////////
// MY CLASS FOR FILE I/O AT LEVEL 1
// automatically closed when it goes out of scope

class MyFile1 {
	HFILE hfile;
public:
	// constructor
	MyFile1(LPCSTR name, UINT mode) {
		OFSTRUCT ofs = {
			sizeof(ofs)
		};

		hfile = OpenFile(name, &ofs, mode);
#ifdef _DEBUG
		if (HFILE_ERROR == hfile)
			MyDebugOutput("OpenFile failed on file1 '%s' with error code %u\n",
				name, ofs.nErrCode);
#endif // _DEBUG
	}
	// destructor
	~MyFile1() {
		if (hfile != HFILE_ERROR)
			_lclose(hfile);
	}
	// access
	operator HFILE() {
		return hfile;
	}
	// logic
	LONG GetLength() {
		LONG here = _llseek(hfile, 0, 1);	// save location
		LONG end = _llseek(hfile, 0, 2);	// seek to end and get length
		_llseek(hfile, here, 0);			// seek back to where i was
		return end;							// return length
	}
};

///////////////////////////////////////////////////////////////////////////////
// MY CLASS FOR FILE I/O AT LEVEL 2
// automatically closed when it goes out of scope

class MyFile2 {
	FILE* file;
	char name[_MAX_PATH];
public:
	// constructor
	MyFile2(LPCSTR fname, LPCSTR mode, size_t bufsiz = BUFSIZ) {
		lstrcpy(name, fname);
		file = fopen(name, mode);
		if (file != NULL) {
			if (bufsiz != BUFSIZ)
				setvbuf(file, NULL, _IOFBF, bufsiz);
		}
		else
			MessageBox(
				hwndParent,						/* handle of parent window	*/
				_strerror(name),				/* address of text in message box	*/
				"PGP Interface",				/* address of title of message box	*/
				MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
	}
	// destructor
	~MyFile2() {
		if (file)
			fclose(file);
	}
	// access
//	operator FILE*() {
//		return file;
//	}
	// logic
	void Close() {
		if (file) {
			fclose(file);
			file = NULL;
		}
	}
	BOOL Eof() {
		return file ? feof(file) : TRUE;
	}
	BOOL Error() {
		return file ? ferror(file) : TRUE ;
	}
	long GetLength() {
		return _filelength(_fileno(file));	// return length
	}
	char* Gets(char* string, int n) {
		char* ptr = fgets(string, n, file);
		
		if (ferror(file))
			MessageBox(
				hwndParent,						/* handle of parent window	*/
				_strerror(name),				/* address of text in message box	*/
				"PGP Interface",				/* address of title of message box	*/
				MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
		return ptr;
	}
	size_t Read(void *buffer, size_t size, size_t count) {
		size_t n = fread(buffer, size, count, file);

		if (ferror(file))
			MessageBox(
				hwndParent,						/* handle of parent window	*/
				_strerror(name),				/* address of text in message box	*/
				"PGP Interface",				/* address of title of message box	*/
				MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
		return n;
	}
	int Puts(char* string) {
		if (file) {
			int ret = fputs(string, file);
			
			if (EOF == ret) {
				MessageBox(
					hwndParent,						/* handle of parent window	*/
					_strerror(name),				/* address of text in message box	*/
					"PGP Interface",				/* address of title of message box	*/
					MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
				Close();
			}
			return ret;
		}
		return EOF;
	}
	int Printf(const char* format, ...) {
		va_list arg;
		int ret;
		
		va_start(arg, format);
		ret = vfprintf(file, format, arg);
		va_end(arg);
		return ret;
	}
	int Flush() {
		return fflush(file);
	}
	int Seek(long offset, int origin) {
		return fseek(file, offset, origin);
	}
	long Tell() {
		return ftell(file);
	}
	int Trunc(long size) {
		return _chsize(_fileno(file), size);
	}
	BOOL Erase() {						// overwrite a file (TRUE if error)
		rewind(file);
		for (long len = GetLength(); len; --len)	// do every byte in file
			putc('\0', file);				// clear it to zero
		Flush();
		if (Error()) {
			MessageBox(
				hwndParent,						/* handle of parent window	*/
				_strerror(name),				/* address of text in message box	*/
				"PGP Interface",				/* address of title of message box	*/
				MB_ICONEXCLAMATION | MB_OK);	/* style of message box	*/
			return TRUE;
		}
		return FALSE;
	}
};

///////////////////////////////////////////////////////////////////////////////

//	PGPJN.H
