
/*  fill_holes.c                             Brian Tierney,  LBL  3/90
 *
 *  fills holes in a binary images
 *
 *  usage: fill_holes [-s NN] < infile > outfile
 *    where -s NN the size of holes to look for ( default = 2)
 */

/*   This program is copyright (C) 1990, Regents  of  the
University  of  California.   Anyone may reproduce this software,
in whole or in part, provided that:
(1)  Any copy  or  redistribution  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;
(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
to this software under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     This software is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
Bug reports or fixes may be sent to the author, who may or may
not act on them as he desires.
*/

/*   Author:  Brian L. Tierney
 *            Lawrence Berkeley Laboratory
 *            Imaging and Distributed Computing Group
 *            email: bltierney@lbl.gov
*/

#include <stdio.h>
#include <sys/types.h>

#include <hipl_format.h>

char     *Progname;

/* #define DEBUG */

#define HOLE_SIZE 3
#define PVAL 255

void      look_for_holes(), fill_holes(), line_fill();

u_char  **image, **new_image;
int       nrow, ncol;
int       hole_size, ends_only = 0;

/*************************************************************/
main(argc, argv)
    int       argc;
    char     *argv[];
{
    register int f;
    struct header hd;

    u_char  **alloc_2d_byte_array();

    Progname = strsave(*argv);

    parse_args(argc, argv);

    read_header(&hd);
    if (hd.pixel_format != PFBYTE)
	perr("image pixel format must be byte");
    update_header(&hd, argc, argv);
    write_header(&hd);

    nrow = hd.rows;
    ncol = hd.cols;

    image = alloc_2d_byte_array(nrow, ncol);
    new_image = alloc_2d_byte_array(nrow, ncol);

    for (f = 0; f < hd.num_frame; f++) {
	read_2d_byte_array(stdin, image, nrow, ncol);

	look_for_holes();

	write_2d_byte_array(stdout, new_image, nrow, ncol);
	bzero((char *) new_image[0], nrow * ncol);
    }
    fprintf(stderr, "%s done. \n\n", argv[0]);
    return (0);
}

/********************************************************************/
void
look_for_holes()
{
    register int i, j;

    for (i = hole_size; i < nrow - hole_size; i++)
	for (j = hole_size; j < ncol - hole_size; j++)
	    if (image[i][j] > 0) {	/* pixel is part of an object */
		new_image[i][j] = image[i][j];
		fill_holes(i, j);
	    }
}

/*************************************************************/
void
fill_holes(i, j)
    int       i, j;
{
    /*
     * NOTE: since we are scanning all pixels, we only need to look for holes
     * in 2 directions, the other 2 directions will be checked later when at
     * another pixel location
     */

    register int i1, j1, i2, j2, jj, ii;

    int       dist;

    dist = 2;			/* start 2 pixels away from point */

    while (dist <= hole_size + 1) {
	i1 = i - dist;
	j1 = j - dist;
	i2 = i + dist;
	j2 = j + dist;
	if (i1 < 0)
	    i1 = 0;
	if (j1 < 0)
	    j1 = 0;
	if (i2 > nrow - 1)
	    i2 = nrow - 1;
	if (j2 > ncol - 1)
	    j2 = ncol - 1;

	for (jj = j1; jj < j2; jj++) {
	    if (image[i2][jj] > 0)
		draw_line(j, i, jj, i2);
	}

	for (ii = i1; ii < i2; ii++) {
	    if (image[ii][j2] > 0)
		draw_line(j, i, j2, ii);
	}
	dist++;
    }
}

/*********************************************************************/
draw_line(x1, y1, x2, y2)
    int       x1, y1, x2, y2;
{
    int       a, b;

    if (ends_only) {
	a = side_cnt(y1, x1);	/* check if 1st point is a end of a line */
	b = corner_cnt(y1, x1);
#ifdef TEST
	if (a > 1 || b > 1)	/* not an end */
#endif
	    if (a > 1 || b > 1 || (a == 1 && b == 1))	/* not an end */
		return;

	a = side_cnt(y2, x2);	/* check if 2nd point is a end of a line */
	b = corner_cnt(y2, x2);
#ifdef TEST
	if (a > 1 || b > 1)	/* not an end */
#endif
	    if (a > 1 || b > 1 || (a == 1 && b == 1))	/* not an end */
		return;
    }
    line_fill(new_image, x1, y1, x2, y2);
}

/*********************************************************************/

int
side_cnt(i, j)
    int       i, j;		/* current array location */
{
    int       cnt = 0;

    if (i > 0)
	if (image[i - 1][j] > 0)
	    cnt++;
    if (j > 0)
	if (image[i][j - 1] > 0)
	    cnt++;
    if (i < nrow - 1)
	if (image[i + 1][j] > 0)
	    cnt++;
    if (j < ncol - 1)
	if (image[i][j + 1] > 0)
	    cnt++;
    return (cnt);
}

/*********************************************************************/
int
corner_cnt(i, j)
    int       i, j;		/* current array location */
{
    int       cnt = 0;

    if (i > 0 && j > 0)
	if (image[i - 1][j - 1] > 0)
	    cnt++;
    if (i < nrow - 1 && j < ncol - 1)
	if (image[i + 1][j + 1] > 0)
	    cnt++;
    if (i < nrow - 1 && j > 0)
	if (image[i + 1][j - 1] > 0)
	    cnt++;
    if (i > 0 && j < ncol - 1)
	if (image[i - 1][j + 1] > 0)
	    cnt++;

    return (cnt);
}

/***************************************************************/
int
count_neighbors(c, r)
    register int c, r;		/* max value returned is 8 */
{
    int       neighbors = 0, tx, ty;

    tx = ncol - 1;
    ty = nrow - 1;

    if (r > 0)
	if (image[r - 1][c] > 0)
	    neighbors++;
    if (r < ty)
	if (image[r + 1][c] > 0)
	    neighbors++;

    if (c > 0)
	if (image[r][c - 1] > 0)
	    neighbors++;
    if (c < tx)
	if (image[r][c + 1] > 0)
	    neighbors++;

    if (r < ty && c < tx)
	if (image[r + 1][c + 1] > 0)
	    neighbors++;

    if (r > 0 && c > 0)
	if (image[r - 1][c - 1] > 0)
	    neighbors++;

    if (r > 0 && c < tx)
	if (image[r - 1][c + 1] > 0)
	    neighbors++;

    if (r < ty && c > 0)
	if (image[r + 1][c - 1] > 0)
	    neighbors++;

    return (neighbors);
}

/********************************************************************/
void
line_fill(buf, x1, y1, x2, y2)	/* Bresenhams's scan conversion algorithm */
    u_char  **buf;
    int       x1, y1, x2, y2;
 /*
  * this code adapted from:   Digital Line Drawing by Paul Heckbert from
  * "Graphics Gems", Academic Press, 1990
  */
{
    int       d, x, y, ax, ay, sx, sy, dx, dy;

    /* absolute value of a */
#ifndef ABS
#define ABS(a)          (((a)<0) ? -(a) : (a))
#endif

    /* take binary sign of a, either -1, or 1 if >= 0 */
#define SGN(a)          (((a)<0) ? -1 : 1)

    if (x1 == x2 && y1 == y2) {
	/* single point, don 't need to scan convert */
	buf[y1][x1] = PVAL;
	return;
    }
    dx = x2 - x1;
    ax = ABS(dx) << 1;
    sx = SGN(dx);

    dy = y2 - y1;
    ay = ABS(dy) << 1;
    sy = SGN(dy);

    x = x1;
    y = y1;
    if (ax > ay) {		/* x dominant */
	d = ay - (ax >> 1);
	for (;;) {
	    buf[y][x] = PVAL;
	    if (x == x2)
		return;
	    if (d >= 0) {
		y += sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    } else {			/* y dominant */

	d = ax - (ay >> 1);
	for (;;) {
	    buf[y][x] = PVAL;
	    if (y == y2)
		return;
	    if (d >= 0) {
		x += sx;
		d -= ay;
	    }
	    y += sy;
	    d += ax;
	}
    }
}


/*********************************************************/
parse_args(argc, argv)
    int       argc;
    char     *argv[];
{
    hole_size = HOLE_SIZE;
    ends_only = 0;

    /* Interpret options  */
    while (--argc > 0 && (*++argv)[0] == '-') {
	char     *s;
	for (s = argv[0] + 1; *s; s++)
	    switch (*s) {
	    case 's':
		if (argc < 2)
		    usageterm();
		sscanf(*++argv, "%d", &hole_size);
		argc--;
		break;
	    case 'e':
		ends_only++;
		break;
	    case 'h':
		usageterm();
		break;
	    default:
		usageterm();
		break;
	    }
    }				/* while */

#ifdef DEBUG
    fprintf(stderr, " looking for holes with size: %d \n", hole_size);
    if (ends_only)
	fprintf(stderr, " looking for holes at ends only  \n");
#endif
}

/*********************************************************************/

usageterm()
{
    fprintf(stderr, "Usage: fill_holes [-s N][-e] < inseq > outseq \n ");
    fprintf(stderr, " Options: [-s N] fill holes of size N ( default = %d) \n", HOLE_SIZE);
    fprintf(stderr, "           [-e] fill holes between ends of lines only \n\n");

    exit(0);
}
