
/* autocollage.c                                 Brian Tierney, LBL   4/90
 *
 *   usage:   autocollage < imseq > outimage
 *
 *  creates a single frame from a sequence of images
 *
 *  Works with data types: Byte only
 */

/*   Author:  Brian L. Tierney
 *            Lawrence Berkeley Laboratory
 *            Imaging Technologies Group
 *            email: bltierney@lbl.gov
*/

#include <stdio.h>
#include <sys/types.h>

#include <hipl_format.h>

#define Calloc(a,b) (b *)calloc((unsigned)(a), sizeof(b))

#define MAX_COLS 2048  /* size for kodak printer: 8 1/2 x 11 */
#define MAX_ROWS 1536

#define GAP 50    /* number of pixels between the images */

#define DEBUG

void      usageterm();
char     *Progname;

/* command line args */
int bgval, outx, outy;

/******************************************************/
main(argc, argv)
    int       argc;
    char    **argv;
{
    struct header hd;		/* hips header */
    int       nframe, nrow, ncol, startx, starty, rw_size;
    int       pic_across, pic_down, out_rows, out_cols, rcnt;
    register int fr, i, j;
    u_char   *image, *iptr;
    u_char  **obuf;

    u_char  **alloc_2d_byte_array();

    Progname = strsave(*argv);
    parse_args(argc, argv);

    read_header(&hd);
    if (hd.pixel_format != PFBYTE) {
	fprintf(stderr, " input must be type byte \n");
	exit(0);
    }
    nframe = hd.num_frame;
    ncol = hd.cols;
    nrow = hd.rows;

    pic_across = outx / (ncol + GAP);
    pic_down = outy / (nrow + GAP);

    if (pic_across * pic_down < nframe) {
	fprintf(stderr, " Error: too many frame in the sequence \n\n");
	exit(0);
    }
    out_cols = pic_across * (ncol + GAP);
    out_cols = ((out_cols / 8) + 1) * 8;  /* round up to multile of 8 */

    out_rows = pic_down * (nrow + GAP);
    out_rows = ((out_rows / 8) + 1) * 8;

    fprintf(stderr, "Creating an image of size %dx%d which has  \n",
	    out_cols, out_rows);
    fprintf(stderr, " %d images across and %d image down with %d pixels between the images \n\n", pic_across, pic_down, GAP);

    image = (u_char *) halloc(nrow * ncol, sizeof(u_char));
    obuf = alloc_2d_byte_array(out_rows, out_cols);

    /* initialize to bgval */
    for(i=0; i< out_rows; i++)
       for(j=0; j< out_cols; j++)
	obuf[i][j] = bgval;

    hd.num_frame = 1;
    hd.cols = out_cols;
    hd.rows = out_rows;
    update_header(&hd, argc, argv);
    write_header(&hd);

    rw_size = nrow * ncol * sizeof(u_char);
    startx = starty = GAP / 2;
    rcnt = 0;

    for (fr = 0; fr < nframe; fr++) {

#ifdef DEBUG
	fprintf(stderr, "frame %d: at location %d, %d \n", fr, startx, starty);
#endif
	if (pread(0, image, rw_size) != rw_size)
	    perr("error during read");

	iptr = image;

	for (i = starty; i < starty + nrow; i++)
	    for (j = startx; j < startx + ncol; j++)
		obuf[i][j] = *iptr++;

	rcnt++;
	if (rcnt < pic_across) {
	    startx = j + GAP;
	} else {
	    rcnt = 0;
	    starty = i + GAP;
	    startx = GAP / 2;
	}
    }

    fprintf(stderr, " writing new collaged image... \n");
    write_2d_byte_array(stdout, obuf, out_rows, out_cols);

    fprintf(stderr, "autocollage done. \n");
    return (0);
}

/****************************************************************/

parse_args(argc, argv)
    int       argc;
    char     *argv[];
{

    /* set defaults */
    bgval = 255;  /* white */
    outx = MAX_COLS;
    outy = MAX_ROWS;

    /* Interpret options  */
    while (--argc > 0 && (*++argv)[0] == '-') {
	char     *s;
	for (s = argv[0] + 1; *s; s++)
	    switch (*s) {
	    case 'b':
                if (argc < 2)
                    usageterm();
                sscanf(*++argv, "%d", &bgval);
                argc--;
                break;
	    case 'o':
                if (argc < 3)
                    usageterm();
                sscanf(*++argv, "%d", &outx);
                argc--;
                sscanf(*++argv, "%d", &outy);
                argc--;
                break;
	    case 'h':
		usageterm();
		break;
	    default:
		usageterm();
		break;
	    }
    }
}

/******************************************************/
void
usageterm()
{
    fprintf(stderr, "Usage: autocollage [-b NN] [-o NN NN] < inseq > outimage  \n");
    fprintf(stderr, "       [-b NN] gray value for background between images (default = 255) \n");
    fprintf(stderr, "       [-o NN NN] maximum size of output image (default = 2048x1536) \n\n");
    exit(0);
}
