/* FileList.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/*
 *
 * FileList -- operations on the list of files.  This is the Document
 * Directory part of NX-Text.
 *
 * $Id: FileList.c,v 1.8 90/10/13 02:39:05 lee Rel1-10 $
 *
 * $Log:	FileList.c,v $
 * Revision 1.8  90/10/13  02:39:05  lee
 * deleted some incorrect code.
 * 
 * Revision 1.7  90/10/13  02:21:03  lee
 * NEEDALIGN stuff
 * 
 * Revision 1.6  90/10/07  20:37:18  lee
 * changed ifdef sparc to ifdef NEEDALIGN
 * 
 * Revision 1.5  90/10/06  00:11:55  lee
 * Prepared for first beta release.
 * 
 * Revision 1.4  90/09/29  23:46:14  lee
 * very minor speedup, and changed a free() to efree().
 * 
 * Revision 1.3  90/09/20  19:11:05  lee
 * deleted unused locking code.
 * removed a sun4-specific memory leak.  Other minor changes.
 * 
 * Revision 1.2  90/08/29  21:46:33  lee
 * Alpha release.
 * 
 * Revision 1.1  90/08/09  19:16:15  lee
 * Initial revision
 * 
 * Revision 2.2  89/10/08  20:29:10  lee
 * Working version of nx-text engine.  Addfile and wordinfo work OK.
 * 
 * Revision 2.1  89/10/02  01:12:08  lee
 * New index format, with Block/WordInBlock/Flags/BytesSkipped info.
 * 
 * Revision 1.2  89/09/16  21:15:54  lee
 * First demonstratable version.
 * 
 * Revision 1.1  89/09/07  21:01:36  lee
 * Initial revision
 * 
 *
 */

#include "globals.h" /* defines and declarations for database filenames */

#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>

#include "smalldb.h"
#include "fileinfo.h"
#include "emalloc.h"

/** Unix system calls that need to be declared: **/
extern int stat();
extern int open(), close(), creat();
extern void exit();
extern int read(), write();
extern unsigned alarm();
/** library functions that need to be declared: */
extern int lockf();
extern unsigned sleep();
extern void perror();
extern long atol();

/** other (lqtext) functions **/
t_FID GetNextFID();
t_FileInfo *GetFileInfo();
/** **/

t_FID
GetMaxFID()
{
    extern int errno;

    int fd;
    struct stat StatBuf;
    char Buffer[20];

    /* ensure that the file is there */
    if (stat(FidFile, &StatBuf) == -1) {
	return 0;
    }

    if ((fd = open(FidFile, O_RDWR, 0)) < 0) {
	fprintf(stderr, "Warning: Can't open FID file");
	return 0;
    }

    /* Read the file */
    if (read(fd, Buffer, (unsigned int) StatBuf.st_size) < 0) {
	fprintf(stderr, "Can't read from \"%s\"\n", FidFile);
	exit(1);
    }

    (void) close(fd);

    Buffer[StatBuf.st_size] = '\0';

    return atol(Buffer);
}

/*ARGSUSED*/
t_FID
GetNextFID(Size)
    long Size; /* to let it keep short FIDs for huge files, execpt I don't */
{
    extern int errno;
    extern long atol();
    extern long lseek();

    int fd;
    char Buffer[21];
    struct stat StatBuf;
    t_FID Result;

    /* ensure that the file is there */
    if (stat(FidFile, &StatBuf) == -1) {
	fprintf(stderr, "Creating FID file \"%s\"\n", FidFile);
	if ((fd = creat(FidFile, 02666)) < 0) {
	    fprintf(stderr, "Can't create FID file \"%s\"\n", FidFile);
	    exit(1);
	}
	(void) close(fd);
	return GetNextFID(Size);

	/*NOTREACHED*/
    }

    if ((fd = open(FidFile, O_RDWR, 0)) < 0) {
	fprintf(stderr, "Can't open FID file");
	perror(FidFile);
	exit(1);
    }

    errno = 0;

    /* Read the file */
    if (read(fd, Buffer, (unsigned int) StatBuf.st_size) < 0) {
	fprintf(stderr, "Can't read from \"%s\"\n", FidFile);
	exit(1);
    }

    Buffer[StatBuf.st_size] = '\0';

    Result = atol(Buffer);

    if (Result == 0L || *Buffer == '-') {
	Result = 1L;
    }

    (void) sprintf(Buffer, "%lu\n", Result + 1);

    /* Move to the start of the file and write the now value.
     * No need to truncate the file, because it didn't shrink!
     */
    (void) lseek(fd, 0, 0L);
    (void) write(fd, Buffer, (unsigned int) strlen(Buffer));
    (void) close(fd);

    return Result;
}

typedef struct {
    t_FID FID;
    time_t DateLastIndexed;
    int FilterType;
    unsigned NameLength;
    char CurrentLocation[1];
} t_PhysicalIndexEntry;

t_PhysicalIndexEntry *
FileInfo2Phys(FileInfo)
    t_FileInfo *FileInfo;
{
    t_PhysicalIndexEntry *PIE;
    register int NameLength;

    if (!FileInfo || !FileInfo->Name) return (t_PhysicalIndexEntry *) 0;

    NameLength = strlen(FileInfo->Name);

    PIE = (t_PhysicalIndexEntry *) emalloc(
				sizeof(t_PhysicalIndexEntry) + NameLength + 1);

    if (!PIE) return (t_PhysicalIndexEntry *) 0;

    PIE->FID = FileInfo->FID;
    PIE->DateLastIndexed = FileInfo->Date;
    PIE->FilterType = FileInfo->FilterType;
    PIE->NameLength = NameLength;
    (void) strcpy(PIE->CurrentLocation, FileInfo->Name);
    return PIE;
}

t_FileInfo *
Phys2FileInfo(PIE)
    t_PhysicalIndexEntry *PIE;
{
    t_FileInfo *FileInfo;

    if (!PIE || !PIE->NameLength) return (t_FileInfo *) 0;

    FileInfo = (t_FileInfo *) emalloc(sizeof(t_FileInfo));
    FileInfo->FID = PIE->FID;
    FileInfo->Date = PIE->DateLastIndexed;
    FileInfo->FilterType = PIE->FilterType;
    FileInfo->Stream = (FILE *) 0;
    if (PIE->NameLength) {
#if 0
	char *doc;
#endif

	FileInfo->Name = emalloc(PIE->NameLength + 1);
	(void) strncpy(FileInfo->Name, PIE->CurrentLocation,
						    PIE->NameLength);
	FileInfo->Name[PIE->NameLength] = '\0';

#if 0
	/* with this in place, wordinfo spends over 40% of its time
	 * in stat!
	 */
	if ((doc = FindFile(FileInfo->Name)) != (char *) 0) {
	    /* hence, we never retrieve non-existent files */
	    FileInfo->Name = erealloc(FileInfo->Name, strlen(doc) + 1);
	    (void) strcpy(FileInfo->Name, doc);
	}
#endif
    } else {
	FileInfo->Name = (char *) 0;
    }

    return FileInfo;
}

int
SaveFileInfo(FileInfo)
    t_FileInfo *FileInfo;
{
    t_PhysicalIndexEntry *PIE;
    datum key, data;
    DBM *db;
    int RetVal;
    char Buffer[20];

    if (!FileInfo) return -1;

    if ((PIE = FileInfo2Phys(FileInfo)) == (t_PhysicalIndexEntry *) 0) {
	return -1;
    }

    if ((db = startdb(FileIndex)) == (DBM *) 0) {
	return -1;
    }

    if (FileInfo->Name && *(FileInfo->Name)) {
	/* For the reverse mapping, FileName --> FID ... store an
	 * entry of the form ([\377]317, "hello").
	 * This scheme simply has to go.
	 * I favour a btree, but that may be needlessly complex.
	 */
	int KeyLen = strlen(FileInfo->Name);
	key.dptr = emalloc(KeyLen + 2); /* +2: "\375" and \0 */
	/* Note: the N= is so that a file called "123" does not cause
	 * confusion with the reverse mapping
	 */
	*(key.dptr) = '\375';
	(void) strcpy(&(key.dptr[1]), FileInfo->Name);
	key.dsize = KeyLen + 1;
		/* length of name + length of "\375" -- the nul at the end
		 * is not included.
		 */

	(void) sprintf(Buffer, "%lu", FileInfo->FID);
	data.dptr = Buffer;
	data.dsize = strlen(Buffer);
	(void) dbm_store(db, key, data, DBM_REPLACE);
	(void) efree(key.dptr);
    }

    (void) sprintf(Buffer, "F%lu", FileInfo->FID);

    key.dptr = Buffer;
    key.dsize = strlen(Buffer);

    data.dptr = (char *) PIE;
    data.dsize = sizeof(t_PhysicalIndexEntry) + PIE->NameLength;

    RetVal = dbm_store(db, key, data, DBM_REPLACE);

    enddb(db);

    return RetVal;
}

t_FID
Name2FID(Name)
    char *Name;
{
    DBM *db;
    datum key, result;
    extern long atol();

    key.dsize = strlen(Name);
    /* see previous routine for comments about this +2 ugliness */
    key.dptr = emalloc(key.dsize + 2);
    *(key.dptr) = '\375';
    (void) strcpy(&(key.dptr[1]), Name);
    key.dsize += 1; /* for the cookie; we don't include the \0 */

    if ((db = startdb(FileIndex)) == (DBM *) 0) {
	fprintf(stderr, "Name2FID can't get FID for %s (database \"%s\"\n", Name, FileIndex);
	(void) efree(key.dptr);
	return -1;
    }
    result = dbm_fetch(db, key);
    enddb(db);

    (void) efree(key.dptr);

    return (result.dsize == 0) ? (t_FID) 0 : atol(result.dptr);
}

t_FileInfo *
GetFileInfo(FID)
    t_FID FID;
{
    t_FileInfo *FileInfo;
    datum key, data;
    DBM *db;
    char Buffer[20];
#ifdef NEEDALIGN
    t_PhysicalIndexEntry *PIE;
#endif

    (void) sprintf(Buffer, "F%lu", FID);
    key.dptr = Buffer;
    key.dsize = strlen(Buffer);

    if ((db = startdb(FileIndex)) == (DBM *) 0) {
	return (t_FileInfo *) 0;
    }

    data = dbm_fetch(db, key);
    enddb(db);

    if (data.dsize == 0) {
	return (t_FileInfo *) 0;
    }

#ifdef NEEDALIGN
    PIE = (t_PhysicalIndexEntry *) emalloc(data.dsize + 1);
    (void) memcpy((char *) PIE, data.dptr, data.dsize);
    FileInfo = Phys2FileInfo(PIE);
    (void) efree((char *) PIE);
#else

    /* Now we have a PIE, so we need a FileInfo... */
    FileInfo = Phys2FileInfo(/*NOSTRICT*/(t_PhysicalIndexEntry *) data.dptr);
#endif

    return FileInfo;
}

int
strcontains(ShortString, LongString)
    char *ShortString;
    char *LongString;
{
    register char *p;

    int strprefix();

    for (p = LongString; *p; p++) {
	if (*p == *ShortString && strprefix(ShortString, p)) {
	    return 1;
	}
    }
    return 0;
}

int
strprefix(Prefix, String)
    register char *Prefix;
    register char *String;
{
    while (*String++ == *Prefix++)
	if (!*Prefix) return 1;
    return 0;
}
