
/*
 * auconfig
 * 	config an audio device
 *
 * usage:
 *	auconfig [-e encoding] [-f freq] [-v vol] audiodevice
 *
 * auconfig sets the per-unit information for a given
 * audio channel.  Settable values are:
 * 	sampling frequency - in KHz.  default 8000
 *      volume             - 0 (soft) to 255 (loud).  default 200 
 *	encoding -             default mu-law
 *		0, ulaw    - 8 bit mu-law
 *              1, alaw    - 8 bit A-law
 *		2, linear  - 8 bit unsigned linear
 *		3, signed  - 8 bit signed linear
 */

#include <fcntl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include "bsd_audioio.h"

#define DEFAULT_ENCODING	AUDIO_ENCODING_ULAW;
#define DEFAULT_VOLUME		200
#define DEFAULT_FREQ		8000

/* encoding string to number       1        2         3        4       */
char           *encoding_str[] = {"ulaw", "alaw", "linear", "signed", 0};

decode_encoding(str)
	char           *str;
{
	int             i;

	/* let ioctl catch bad encoding numbers */
	if (isdigit(str[0]))
		return (atoi(str));

	/* string -> number */
	for (i = 0; encoding_str[i]; i++) {
		if (strcmp(str, encoding_str[i]) == 0)
			return (i + 1);
	}

	printf("Unknown encoding %s\n", str);
	printf("valid encodings: \n");
	for(i = 0; encoding_str[i]; i++) {
		printf("\t%s", encoding_str[i]);
	}
	printf("\n");
	exit(1);
}

usage(name)
	char           *name;
{
	printf("usage:  %s [-e encoding] [-f freq] [-v volume] device ...\n", 
		name);
	exit(1);
}

main(argc, argv)
	char          **argv;
{
	audio_info_t    ai;
	char           *name, *file;
	int             fd, c;
	extern int      optind;
	extern char    *optarg;

	bzero(&ai, sizeof(ai));
	ai.mode = AUMODE_PLAY;
	ai.play.sample_rate = DEFAULT_FREQ;
	ai.play.gain = DEFAULT_VOLUME;
	ai.play.encoding = DEFAULT_ENCODING;

	name = argv[0];
	while ((c = getopt(argc, argv, "e:f:v:")) > 0) {
		switch (c) {
		case 'e':
			ai.play.encoding = decode_encoding(optarg);
			break;
		case 'f':
			ai.play.sample_rate = atoi(optarg);
			break;
		case 'v':
			ai.play.gain = atoi(optarg);
			break;
		default:
			printf("bad option %c\n", c);
			usage(name);
		}
	}
	argc -= optind;
	argv += optind;

	if (argc < 1) {
		printf("argc %d\n", argc);
		printf("must have at least one filename\n");
		usage(name);
	}
	while (argc--) {
		file = *argv++;

		fd = open(file, O_WRONLY);
		if (fd < 0) {
			perror(file);
			continue;
		}
		if (ioctl(fd, AUDIO_SETINFO, &ai) < 0)
			perror("ioctl");

		close(fd);
	}
}

