/*********************************************************************
**
**     File name:               ssh_packet.c
**
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 pack and unpack a message
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**
**     References:              draft-ylonen-ssh-protocol-00.txt
**
**     Notes:
**	The functions in this file pack and unpack a packet.  The
**	packet protocol is described in draft-ylonen-ssh-protocol-00.txt
**	for SSHv1.5.  Basically, each packet contains a
**		packet length (length of packet - [padding + this field])
**		padding	(8 - length % 8)
**		packet type/message type
**		data
**		crc
**	So the functions in this file take the data and put it into packet
**	form and visa versa.
**
**     Functions:
**	ssh_packet_unpack			unpack a packet
**	ssh_packet_pack				pack a packet
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/cia/kohno/libssh/libssh/RCS/ssh_packet.c,v 3.3 1998/03/14 20:58:53 kohno Exp $";
#endif

#include <string.h>
#include <stdlib.h>

#include <rand.h>

#include "ssh.h"
#include "ssh_debug.h"
#include "ssh_packet.h"
#include "ssh_crypt.h"

#include "ssh_util.h"

/*********************************************************************
**
**     Function:                ssh_packet_unpack
**
**     Purpose:                 extract the data from a packet.
**
**     Entry (pre) conditions:  memory for *data allocate
**				pointers valid
**
**     Parameters:              ssh_info	info about current connection
**
**				packet		packet to unpack
**
**				*data_len	length of data
**				*type		type of message
**				*data		data in packet
**
**				out_size	available mem for data
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for
**					SSH_ERRNO_LOW_MEM
**					SSH_ERRNO_BAD_CRC
**				if not enough memory for *data, or
**				there's a bad CRC.
**
**     Side effects:            *type set to packet type
**				*data_len set to actual length of
**				data (just data, not including type, crc)
**				*data holds data in packet (not including
**				type, crc)
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**
**     Notes:
**
*********************************************************************/

int ssh_packet_unpack
(
	struct ssh_struct * ssh_info,	/* info about current connection */

	uint8_t * packet,	/* packet to unpack */

	uint32_t * data_len,	/* length of data */
	uint8_t * type,		/* type of message */
	uint8_t * data,		/* data in packet */

	uint32_t out_size	/* size malloc'ed for * data */
)
{
	uint8_t * packet_ptr;	/* pointer into packet */
	uint32_t packet_length;	/* length of packet */
	uint32_t crc;		/* crc of packet */
	int padding;		/* padding length */

	/*
	**	First lets computer the packet length and the padding
	*/
	my_bcopy((void *) packet, (void *) &packet_length,
		SSH_PACKET_LENGTH_LEN);
	packet_length = ntohl(packet_length);

	padding = SSH_UNITS - packet_length % SSH_UNITS;

	/*
	**	Now we know that the actual data is
	**	SSH_PACKET_LENGTH_LEN + padding bytes into packet
	*/
	*type = packet[SSH_PACKET_LENGTH_LEN + padding];

	/*
	**	Now make sure we can have enough output space
	*/
	*data_len = packet_length - SSH_DATA_LEN_DIFF;

	if (*data_len > out_size)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}

	/*
	**	copy data to user space (floating 1 is for type)
	*/
	packet_ptr = packet + SSH_PACKET_LENGTH_LEN + padding + 1;
	my_bcopy((void *) packet_ptr, (void *) data, *data_len);

	/*
	**	Finally, lets check the CRC
	*/
	packet_ptr = packet + SSH_PACKET_LENGTH_LEN + padding + 1 
		+ *data_len;
	my_bcopy((void *) packet_ptr, (void *) &crc, SSH_PACKET_CRC_LEN);
	crc = ntohl(crc);

	if (crc !=
		crc32(packet + SSH_PACKET_LENGTH_LEN, *data_len + 1 + padding))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"bad crc", "ssh_packet_unpack");

		ssh_errno_set(SSH_ERRNO_BAD_CRC);
		return(S_BAD);
	}

	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                ssh_packet_pack
**
**     Purpose:                 make a packet given the type and data
**
**     Entry (pre) conditions:  memory for packet allocated
**				pointers valid
**
**     Parameters:              *packet		output packet
**				*total_len	total length of *packet
**
**				type		message type to pack
**				data		data for the packet
**				data_len	length of "data"
**
**				out_size	available mem for *packet
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for SSH_ERRNO_LOW_MEM
**				if not enought memory for packet
**
**     Side effects:            an SSHv1.5 packet is formed from the type
**				and data given.  the resulting packet
**				is stored to *packet, and *total_len
**				contains the length of *packet.
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					(just a call to RAND_bytes)
**
**     References:                  
**
**     Notes:
**	buffer overflow checking should be everywhere -- packet formation
**	especially xxx
**
*********************************************************************/
int ssh_packet_pack
(
	uint8_t * packet,	/* packet to pack */
	uint32_t * total_len,	/* total length of output */
	uint8_t type,		/* type of message */
	uint8_t * data,		/* data for message */
	uint32_t data_len,	/* length of data */
	uint32_t out_size	/* valid packet size max */
)	
{
	uint8_t * packet_ptr;	/* pointer to packet */
	uint32_t packet_length;	/* length of packet (type, data, crc) */
	uint32_t nbo_packet_length;	/* packet length in network order */
	uint32_t padding;		/* bytes for padding */
	uint32_t crc;			/* packet's crc */

	packet_length = data_len + SSH_DATA_LEN_DIFF;
	padding = SSH_UNITS - packet_length % SSH_UNITS;

	*total_len = packet_length + padding + SSH_PACKET_LENGTH_LEN;

	if (*total_len > out_size)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}


	/*
	**	now lets fill the packet
	*/
	nbo_packet_length = htonl(packet_length);
	my_bcopy((void *) &nbo_packet_length, (void *) packet,
		SSH_PACKET_LENGTH_LEN);

	/*
	**	fill padding with random bytes
	*/

	RAND_bytes(packet + SSH_PACKET_LENGTH_LEN, padding);

	/*
	**	set type
	*/
	packet[SSH_PACKET_LENGTH_LEN + padding] = type;

	/*
	**	copy in the data (floating 1 for type)
	*/
	packet_ptr = packet + SSH_PACKET_LENGTH_LEN + padding + 1;
	my_bcopy((void *) data, (void *) packet_ptr, data_len);

	/*
	**	set the crc
	*/
	packet_ptr = packet + SSH_PACKET_LENGTH_LEN + padding + 1 + data_len;

	crc = crc32(packet + SSH_PACKET_LENGTH_LEN, data_len + 1 + padding);
	crc = htonl(crc);

	my_bcopy((void *) &crc, (void *) packet_ptr, SSH_PACKET_CRC_LEN);
	
	return(S_GOOD);
}

